<?php

if ( ! class_exists( 'DropboxClient' ) ) {

	class DropboxClient {

		const MAX_UPLOAD_CHUNK_SIZE = 150000000; // 150MB

		const UPLOAD_CHUNK_SIZE = 4000000; // 4MB

		private $appParams;

		private $accessToken;

		function __construct( $app_params, $accessToken ) {
			if ( empty( $app_params['app_key'] ) ) {
				throw new DropboxException( "App Key is empty!" );
			}
			if ( empty( $accessToken ) ) {
				throw new DropboxException( "Access Token is empty!" );
			}
			$this->appParams   = $app_params;
			$this->accessToken = $accessToken;
		}

		// ##################################################
		// API Functions

		public function GetCurrentAccount() {
			return $this->apiCall('users/get_current_account');
		}

		public function CreateFolder($name = '') {
			$folder = $this->GetFileFolder('/' . $name);
			if (isset($folder['error'])) {
				return $this->apiCall('files/create_folder', array(
					'path'       => '/' . $name,
					'autorename' => true
				));
			} else {
				return false;
			}
		}

		public function ListFolder($name = '') {
			$folder = $this->GetFileFolder('/' . $name);
			if (!isset($folder['error'])) {
				return $this->apiCall('files/list_folder', array(
					'path'                                => '/' . $name,
					'recursive'                           => false,
					'include_media_info'                  => false,
					'include_deleted'                     => false,
					'include_has_explicit_shared_members' => false,
				));
			} else {
				return false;
			}
		}

		public function GetFileFolder($path = '') {
			return $this->apiCall('files/get_metadata', array(
				'path'                                => $path,
				'include_media_info'                  => false,
				'include_deleted'                     => false,
				'include_has_explicit_shared_members' => false,
			));
		}

		public function GetLink($path = '') {
			return $this->apiCall('files/get_temporary_link', array(
				'path' => $path
			));
		}

		public function Delete($path = '') {
			$folder = $this->GetFileFolder($path);
			if (isset($folder['error'])) {
				return false;
			} else {
				return $this->apiCall('files/delete', array(
					'path' => $path
				));
			}
		}

		public function Upload($source_file = '', $remote_file = '') {
			// Get the filename
			$filename = basename($source_file);

			// Check if we're uploading to folder or a file
			$remoteLocation = $this->GetFileFolder($remote_file);
			if ($remoteLocation['.tag'] == 'folder') {
				$remote_file = $remoteLocation['path_lower'] . '/' . $filename;
			}

			// Determine if it needs to be chunked down
			$file_size = filesize($source_file);
			if ($file_size > DropboxClient::MAX_UPLOAD_CHUNK_SIZE) {

				// Chunk it down baby!
				$file_pointer = fopen( $source_file, 'rb' );
				$file_session = NULL;
				$file_offset  = 0;
				while ( ! feof( $file_pointer ) ) {
					$chunk        = fread( $file_pointer, self::UPLOAD_CHUNK_SIZE );
					if ($file_session === NULL) {
						$session = $this->uploadCall('files/upload_session/start', array(
							'close' => false
						), $chunk);
						$file_session = $session['session_id'];
					} else {
						$this->uploadCall('files/upload_session/append_v2', array(
							'cursor' => array(
								'session_id' => $file_session,
								'offset'     => $file_offset
							),
							'close'  => false
						), $chunk);
					}
					$file_offset += strlen( $chunk );
					if ( $file_offset >= $file_size ) {
						break;
					}
				}
				// Finish the chunk upload session
				return $this->uploadCall('files/upload_session/finish', array(
					'cursor' => array(
						'session_id' => $file_session,
						'offset'     => $file_offset
					),
					'commit' => array(
						'path'       => $remote_file,
						'mode'       => 'add',
						'autorename' => true,
						'mute'       => false
					)
				), '');

			} else {

				// Regular upload
				$handle   = fopen($source_file, "r");
				$contents = fread($handle, filesize($source_file));

				$output = $this->uploadCall('files/upload', array(
					'path'       => $remote_file,
					'mode'       => 'add',
					'autorename' => true,
					'mute'       => false
				), $contents);

				fclose($handle);

				return $output;
			}
		}

		public function Download($remote_file = '', $source_file = '') {
			return $this->downloadCall('files/download', array(
				'path' => $remote_file
			), $source_file);
		}

		// ##################################################
		// Helper Functions

		private function downloadCall( $path, $fields = null, $file = '' ) {
			$url = $this->cleanUrl( 'https://content.dropboxapi.com/2/' . $path );

			$curl = curl_init();
			curl_setopt($curl, CURLOPT_URL, $url);
			curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
			curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
			curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
			curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);
			curl_setopt($curl, CURLOPT_BINARYTRANSFER, true );

			curl_setopt($curl, CURLOPT_HTTPHEADER, array(
				'Authorization: Bearer ' . $this->accessToken,
				'Dropbox-API-Arg: ' . json_encode($fields)
			));

			curl_setopt($curl, CURLOPT_CUSTOMREQUEST, "POST");

			$file_pointer = @fopen( $file, 'wb' );
			curl_setopt($curl, CURLOPT_FILE, $file_pointer );

			$server_response = curl_exec($curl);

			curl_close($curl);
			fclose( $file_pointer );
			return json_decode($server_response, true);
		}

		private function uploadCall( $path, $fields = null, $file = '' ) {
			@ini_set("memory_limit", "-1");
			set_time_limit(0);
			$url = $this->cleanUrl( 'https://content.dropboxapi.com/2/' . $path );

			$curl = curl_init();
			curl_setopt($curl, CURLOPT_URL, $url);
			curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
			curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
			curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
			curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);
			curl_setopt($curl, CURLOPT_BINARYTRANSFER, true );

			curl_setopt($curl, CURLOPT_HTTPHEADER, array(
				'Authorization: Bearer ' . $this->accessToken,
				'Dropbox-API-Arg: ' . json_encode($fields),
				'Content-Type: application/octet-stream'
			));

			curl_setopt($curl, CURLOPT_CUSTOMREQUEST, "POST");
			curl_setopt($curl, CURLOPT_POSTFIELDS, $file);

			$server_response = curl_exec($curl);

			curl_close($curl);
			return json_decode($server_response, true);
		}

		private function apiCall( $path, $fields = null ) {
			$url = $this->cleanUrl( 'https://api.dropboxapi.com/2/' . $path );

			$curl = curl_init();
			curl_setopt($curl, CURLOPT_URL, $url);
			curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
			curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
			curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
			curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);

			curl_setopt($curl, CURLOPT_HTTPHEADER, array(
				'Authorization: Bearer ' . $this->accessToken,
				'Content-Type: application/json'
			));

			curl_setopt($curl, CURLOPT_CUSTOMREQUEST, "POST");
			curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($fields));

			$server_response = curl_exec($curl);

			curl_close($curl);
			return json_decode($server_response, true);
		}

		private function cleanUrl( $url ) {
			$p   = substr( $url, 0, 8 );
			$url = str_replace( '//', '/', str_replace( '\\', '/', substr( $url, 8 ) ) );
			$url = rawurlencode( $url );
			$url = str_replace( '%2F', '/', $url );

			return $p . $url;
		}
	}

	class DropboxException extends Exception {

		public function __construct( $err = null, $isDebug = false ) {
			if ( is_null( $err ) ) {
				$el            = error_get_last();
				$this->message = $el['message'];
				$this->file    = $el['file'];
				$this->line    = $el['line'];
			} else {
				$this->message = $err;
			}
			self::log_error( $err );
			if ( $isDebug ) {
				self::display_error( $err, true );
			}
		}

		public static function log_error( $err ) {
			error_log( $err, 0 );
		}

		public static function display_error( $err, $kill = false ) {
			print_r( $err );
			if ( $kill === false ) {
				die();
			}
		}
	}

}