<?php

if ( ! class_exists( 'GoogleDriveClient' ) ) {

	class GoogleDriveClient {

		const TOKEN_URL = 'https://www.googleapis.com/oauth2/v4/token';

		private $clientId;

		private $clientSecret;

		private $accessToken;

		function __construct( $app_params, $token ) {

			if ( empty( $app_params['app_key'] ) ) {
				throw new GoogleDriveException( "App Key is empty!" );
			}
			if ( empty( $app_params['app_secret'] ) ) {
				throw new GoogleDriveException( "App Secret Key is empty!" );
			}

			$this->clientId        = $app_params['app_key'];
			$this->clientSecret    = $app_params['app_secret'];

			$this->accessToken  = $token;

		}

		// ##################################################
		// Authorization

		private function RefreshAccessToken () {
			$url = self::TOKEN_URL;
			$curl = curl_init();
			$fields = http_build_query(
				[
					'client_id'     => $this->clientId,
					'client_secret' => $this->clientSecret,
					'refresh_token' => $this->accessToken['refresh_token'],
					'grant_type'    => 'refresh_token',
				]
			);
			curl_setopt_array($curl, [
				// General options.
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_FOLLOWLOCATION => true,
				CURLOPT_AUTOREFERER    => true,
				CURLOPT_POST           => 1,
				CURLOPT_POSTFIELDS     => $fields,
				CURLOPT_HTTPHEADER => [
					'Content-Length: ' . strlen($fields),
				],
				// SSL options.
				CURLOPT_SSL_VERIFYHOST => false,
				CURLOPT_SSL_VERIFYPEER => false,
				CURLOPT_URL            => $url,
			]);
			$result = curl_exec($curl);
			if (false === $result) {
				if (curl_errno($curl)) {
					throw new GoogleDriveException('curl_setopt_array() failed: ' . curl_error($curl));
				} else {
					throw new GoogleDriveException('curl_setopt_array(): empty response');
				}
			}
			$decoded = json_decode($result, true);
			if (null === $decoded) {
				throw new GoogleDriveException('json_decode() failed');
			}

			return $decoded['access_token'];
		}

		// #################################################
		// Communication

		private function apiCall( $path, $method = 'GET', $fields = null ) {
			$url = 'https://www.googleapis.com/drive/v2/' . $path ;

			$curl = curl_init();
			curl_setopt($curl, CURLOPT_URL, $url);
			curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
			curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
			curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
			curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);

			// Get a new Access Token
			$token = $this->RefreshAccessToken();

			curl_setopt($curl, CURLOPT_HTTPHEADER, array(
				'Authorization: Bearer ' . $token,
				'Content-Type: application/json '
			));

			curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $method);

			if ($method === 'POST') {
				curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($fields));
			}

			$server_response = curl_exec($curl);

			curl_close($curl);
			return json_decode($server_response, true);
		}

		private function uploadCall( $path, $fields, $source_file ) {
			@ini_set("memory_limit", "-1");
			set_time_limit(0);
			$url = 'https://www.googleapis.com/upload/drive/v2/' . $path . '?uploadType=multipart' ;

			// Stuff
			$boundary    = '-------314159265358979323846';
			$delimiter   = "\r\n--" . $boundary . "\r\n";
			$close_delim = "\r\n--" . $boundary . "--";

			$curl = curl_init();
			curl_setopt($curl, CURLOPT_URL, $url);
			curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
			curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
			curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
			curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);

			// Get a new Access Token
			$token = $this->RefreshAccessToken();

			// File to Base64
			$base64Data = base64_encode(file_get_contents($source_file));

			$multipartRequestBody =
				$delimiter .
				"Content-Type: application/json\r\n\r\n" .
				json_encode($fields) .
				$delimiter .
				"Content-Type: " . $fields['mimeType'] . "\r\n" .
				"Content-Transfer-Encoding: base64\r\n" .
				"\r\n" .
				$base64Data .
				$close_delim;

			curl_setopt($curl, CURLOPT_HTTPHEADER, array(
				'Authorization: Bearer ' . $token,
				'Content-Type: multipart/mixed; boundary="' . $boundary . '"'
			));


			curl_setopt($curl, CURLOPT_POST, true);
			curl_setopt($curl, CURLOPT_POSTFIELDS, $multipartRequestBody);

			$server_response = curl_exec($curl);

			curl_close($curl);
			return json_decode($server_response, true);
		}

		private function downloadCall( $id = 0, $file = '' ) {
			$url = 'https://www.googleapis.com/drive/v2/files/' . $id . '?alt=media';

			$curl = curl_init();
			curl_setopt($curl, CURLOPT_URL, $url);
			curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
			curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
			curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
			curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);
			curl_setopt($curl, CURLOPT_BINARYTRANSFER, true );

			// Get a new Access Token
			$token = $this->RefreshAccessToken();

			curl_setopt($curl, CURLOPT_HTTPHEADER, array(
				'Authorization: Bearer ' . $token
			));

			$file_pointer = @fopen( $file, 'wb' );
			curl_setopt($curl, CURLOPT_FILE, $file_pointer );

			$server_response = curl_exec($curl);

			curl_close($curl);
			fclose( $file_pointer );
			return json_decode($server_response, true);
		}

		// #################################################
		// APIs

		public function Delete($id = 0) {
			return $this->apiCall('files/' . $id, 'DELETE');
		}

		public function ListFolders() {
			return $this->apiCall('files?trashed=false&q=mimeType=\'application/vnd.google-apps.folder\'');
		}

		public function ListFiles( $query ) {
			$folder = $this->GetFolder($query);
			return $this->apiCall('files?trashed=false&q="'.$folder.'"+in+parents+and+mimeType+!=+"application/vnd.google-apps.folder"');
		}

		public function FindFiles( $query ) {
			return $this->apiCall('files?trashed=false&q=title+contains+\''.$query.'\'+and+mimeType+!=+\'application/vnd.google-apps.folder\'');
		}

		private function GetFolder( $name ) {
			$folders = $this->ListFolders();
			foreach($folders['items'] as $folder) {
				if ($folder['title'] === $name) {
					return $folder['id'];
				}
			}
			return false;
		}

		public function CreateFolder( $name, $parent = null ) {
			if ($this->GetFolder($name) === false) {
				$data = array(
					'title'    => $name,
					'mimeType' => 'application/vnd.google-apps.folder'
				);
				if ($parent !== null) {
					$parent = $this->GetFolder($parent);
					if ($parent !== false) {
						$data['parents'] = array(array('id'=>$parent));
					}
				}
				return $this->apiCall('files', 'POST', $data);
			} else {
				return false;
			}
		}

		public function Upload( $source_file, $folder ) {
			$folder_id = $this->GetFolder($folder);
			if ($folder_id !== false) {
				$data = array(
					'title'      => basename($source_file),
					'mimeType'   => 'application/zip',
					'parents'    => array( array('id'=>$folder_id) )
				);
				return $this->uploadCall('files', $data, $source_file);
			} else {
				return false;
			}
		}

		public function Download( $remote_file, $source_file ) {
			return $this->downloadCall($remote_file, $source_file);
		}

	}

	class GoogleDriveException extends Exception {

		public function __construct( $err = null, $isDebug = false ) {
			if ( is_null( $err ) ) {
				$el            = error_get_last();
				$this->message = $el['message'];
				$this->file    = $el['file'];
				$this->line    = $el['line'];
			} else {
				$this->message = $err;
			}
			self::log_error( $err );
			if ( $isDebug ) {
				self::display_error( $err, true );
			}
		}

		public static function log_error( $err ) {
			error_log( $err, 0 );
		}

		public static function display_error( $err, $kill = false ) {
			print_r( $err );
			if ( $kill === false ) {
				die();
			}
		}
	}

}