<?php

if ( ! class_exists( 'OnedriveClient' ) ) {
    class OnedriveClient {

        /** App Limitations **/
        const MAX_UPLOAD_CHUNK_SIZE = 60000000; // 150MB
        const UPLOAD_CHUNK_SIZE = 4000000; // 4MB

        /** App Setting **/
        private $appParams;
        private $accessToken;

        /** Microsoft Graph root URL and version **/
        const GRAPH_URL = "https://graph.microsoft.com/v1.0";

        /** Authorization URL **/
        const AUTH_URL = "https://login.microsoftonline.com/common/oauth2/v2.0/";



        function __construct( $app_params ) {
            $this->appParams = $app_params;

            if ( empty( $app_params['client_id'] ) ) {
                throw new OnedriveException( "Client ID is empty!" );
            }
            if ( empty( $app_params['client_secret'] ) ) {
                throw new OnedriveException( "Client Secret is empty!" );
            }

            $this->accessToken  = null;
        }


        /** Authorization **/
        public function SetAccessToken ($token)
        {
            $this->accessToken = $token;
        }

        public function BuildAuthorizeUrl()
        {
            $url = self::AUTH_URL . 'authorize'
                . '?client_id=' . urlencode($this->appParams['client_id'])
                . '&scope=' . urlencode("offline_access files.readwrite")
                . '&response_type=code'
                . '&redirect_uri=' . urlencode($this->appParams['redirect_uri']);
            return $url;
        }

        public function BuildDeAuthorizeUrl()
        {
            $url = self::AUTH_URL . 'logout'
                . '?post_logout_redirect_uri=' . urlencode($this->appParams['redirect_uri']);
            return $url;
        }

        public function obtainAccessToken($code)
        {
            if (null === $this->appParams['client_id']) {
                throw new OnedriveException( 'The client ID must be set to call obtainAccessToken()' );
            }
            if (null === $this->appParams['redirect_uri']) {
                throw new OnedriveException( 'The state\'s redirect URI must be set to call obtainAccessToken()' );
            }

            $at = $this->authCall('token', array(
                'client_id'     => $this->appParams['client_id'],
                'redirect_uri'  => $this->appParams['redirect_uri'],
                'client_secret' => $this->appParams['client_secret'],
                'code'          => $code,
                'grant_type'    => 'authorization_code'
            ));

            if ( empty( $at ) ) {
                throw new OnedriveException( 'Could not get access token!');
            }

            if (isset($at['error'])) {
                throw new OnedriveException( $at['error_description'] );
            }

            return ( $this->accessToken = $at );
        }

        public function renewAccessToken()
        {
            if (null === $this->appParams['client_id']) {
                throw new OnedriveException('The client ID must be set to call renewAccessToken()');
            }
            if (null === $this->accessToken['refresh_token']) {
                throw new OnedriveException('The refresh token is not set or no permission for \'wl.offline_access\' was given to renew the token');
            }

            $at = $this->authCall('token', array(
                'client_id'     => $this->appParams['client_id'],
                //'redirect_uri'  => $this->appParams['redirect_uri'],
                'client_secret' => $this->appParams['client_secret'],
                'refresh_token' => $this->accessToken['refresh_token'],
                'grant_type'    => 'refresh_token'
            ));

            if ( empty( $at ) ) {
                throw new OnedriveException( 'Could not get access token!');
            }

            if (isset($at['error'])) {
                throw new OnedriveException( $at['error_description'] );
            }

            return ( $this->accessToken = $at );
        }

        /** API Functions **/
        public function CreateFolder()
        {
            $folder = $this->apiCall('/drive/root:/psv3:/children', 'GET');
            if ( isset($folder['error']) ) {
                return $this->apiCall('/drive/root/children', 'POST', array(
                    "name" => "psv3",
                    "folder" => array()
                ));
            } else {
                return false;
            }
        }

        public function GetFileFolder($location)
        {
            return $this->apiCall($location, 'GET');
        }

        public function upload($source_file = '')
        {
            // Get the filename
            $filename = basename($source_file);

            // Determine if it needs to be chunked down
            $file_size = filesize($source_file);

            // Chunk it down baby!
            $file_pointer = fopen( $source_file, 'rb' );
            $file_session = NULL;
            $file_offset  = 0;

            $session = $this->apiCall('/drive/root:/psv3/'.$filename.':/createUploadSession', 'POST', array(
                "item"  => array(
                    "@microsoft.graph.conflictBehavior" => "rename",
                    "name" => $filename
                )
            ));
            $file_session = $session;

            $output = null;

            while ( ! feof( $file_pointer ) ) {

                $chunk = fread( $file_pointer, self::UPLOAD_CHUNK_SIZE );

                $fragment_size = $file_offset + strlen( $chunk ) - 1;
                $headers = array(
                    "Content-Length: " . strlen( $chunk ),
                    "Content-Range: bytes " . $file_offset . '-' . $fragment_size . '/' . $file_size
                );

	            $output = $this->uploadCall($file_session['uploadUrl'], $headers, $chunk);

                $file_offset += strlen( $chunk );

                if ( $file_offset >= $file_size ) {
                    break;
                }
            }

            $this->cancelUploadSession($file_session['uploadUrl']);
            return $output;
        }

        /** Helper Functions **/

        private function authCall( $path, $request_data = null )
        {
            $url     = self::AUTH_URL . $path;

            $curl = curl_init();
            curl_setopt($curl, CURLOPT_URL, $url);
            curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 400);
            curl_setopt($curl, CURLOPT_TIMEOUT, 400);
            curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);

            curl_setopt($curl, CURLOPT_HTTPHEADER, array(
                'Content-Type: application/x-www-form-urlencoded'
            ));

            curl_setopt($curl, CURLOPT_HEADER, false);
            curl_setopt($curl, CURLOPT_COOKIEFILE, "");
            curl_setopt($curl, CURLOPT_ENCODING, "gzip, deflate");
            curl_setopt($curl, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4 );
            curl_setopt($curl, CURLOPT_USERAGENT, 'Xagio Remote Admin Panel (version 3) - Secure Connection');

            curl_setopt($curl, CURLOPT_POST, 1);
            curl_setopt($curl, CURLOPT_POSTFIELDS, http_build_query($request_data));

            $server_response = curl_exec($curl);

            curl_close($curl);
            return json_decode($server_response, true);
        }

        public function apiCall( $path, $method = "GET", $fields = null )
        {
            $url = self::GRAPH_URL . $path;

            $curl = curl_init();
            curl_setopt($curl, CURLOPT_URL, $url);
            curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);

            curl_setopt($curl, CURLOPT_HTTPHEADER, array(
                'Authorization: Bearer ' . $this->accessToken['access_token'],
                'Content-Type: application/json'
            ));

            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $method);
            if ($method == "POST") {
                curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($fields, JSON_FORCE_OBJECT));
            }

            $server_response = curl_exec($curl);

            curl_close($curl);
            return json_decode($server_response, true);
        }

        public function uploadCall($url, $headers, $file)
        {
	        @ini_set("memory_limit", "-1");
	        set_time_limit(0);
            $curl = curl_init();
            curl_setopt($curl, CURLOPT_URL, $url);
            curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);

            curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);

            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'PUT');
            curl_setopt($curl, CURLOPT_POSTFIELDS, $file);

            $server_response = curl_exec($curl);

            curl_close($curl);
            return json_decode($server_response, true);
        }

        public function cancelUploadSession($url)
        {
            $curl = curl_init();
            curl_setopt($curl, CURLOPT_URL, $url);
            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'DELETE');
            $server_response = curl_exec($curl);
            //$httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE); //
            curl_close($curl);
            return json_decode($server_response, true);
        }

        public function downloadCall($path, $file = '')
        {
            $url = self::GRAPH_URL . $path;

            $curl = curl_init();
            curl_setopt($curl, CURLOPT_URL, $url);
            curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($curl, CURLOPT_BINARYTRANSFER, true );

            curl_setopt($curl, CURLOPT_HTTPHEADER, array(
                'Authorization: Bearer ' . $this->accessToken['access_token'],
                'Content-Type: application/json'
            ));

            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, "GET");

            $file_pointer = @fopen( $file, 'wb' );
            curl_setopt($curl, CURLOPT_FILE, $file_pointer );

            $server_response = curl_exec($curl);

            curl_close($curl);
            fclose( $file_pointer );
            return json_decode($server_response, true);
        }

        public function deleteCall($path)
        {
            $url = self::GRAPH_URL . $path;

            $curl = curl_init();
            curl_setopt($curl, CURLOPT_URL, $url);
            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'DELETE');

            curl_setopt($curl, CURLOPT_HTTPHEADER, array(
                'Authorization: Bearer ' . $this->accessToken['access_token'],
                'Content-Type: application/json'
            ));

            $server_response = curl_exec($curl);
            $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            curl_close($curl);
            return $httpCode;
        }

    }

    class OnedriveException extends Exception {

        public function __construct( $err = null, $isDebug = false ) {
            if ( is_null( $err ) ) {
                $el            = error_get_last();
                $this->message = $el['message'];
                $this->file    = $el['file'];
                $this->line    = $el['line'];
            } else {
                $this->message = $err;
            }
            self::log_error( $err );
            if ( $isDebug ) {
                self::display_error( $err, true );
            }
        }

        public static function log_error( $err ) {
            error_log( $err, 0 );
        }

        public static function display_error( $err, $kill = false ) {
            print_r( $err );
            if ( $kill === false ) {
                die();
            }
        }
    }
}