<?php

if (!class_exists('MXAG_Api')) {

	class MXAG_Api
	{

		public static function initialize()
		{
			// Generate API if there's none
			self::generateAPIKey();

			// Add the API gateway
			add_action('admin_post_nopriv_prs_api', ['MXAG_Api', 'handleRequest']);
			add_action('admin_post_prs_api', ['MXAG_Api', 'handleRequest']);

			// Remote Login gateway
			add_action('admin_post_nopriv_prs_remote_login', ['MXAG_Api', 'remoteLogin']);
			add_action('admin_post_prs_remote_login', ['MXAG_Api', 'remoteLogin']);

			// Plugin Triggers
			add_action('deleted_plugin', ['MXAG_Api', 'pluginDeleted'], 10, 2);
			add_action('deactivated_plugin', ['MXAG_Api', 'pluginDeactivated'], 10, 2);
			add_action('activated_plugin', ['MXAG_Api', 'pluginActivated'], 10, 2);

			// Theme Trigger
			add_action('after_switch_theme', ['MXAG_Api', 'themeSwitch']);
		}


		/***************************************************************************************************************
		 *
		 *   Triggers
		 *
		 **************************************************************************************************************/

		public static function syncWithPanel()
		{
			MXAG_Api::apiRequest(
				$endpoint = 'sync',
				$method = 'GET',
				[
					'domain' => preg_replace('/^www\./', '', $_SERVER['SERVER_NAME'])
				]
			);
		}

		public static function themeSwitch($old_name)
		{
			$new_theme = wp_get_theme();
			MXAG_Api::apiRequest(
				$endpoint = 'themes',
				$method = 'POST',
				[
					'domain' => preg_replace('/^www\./', '', $_SERVER['SERVER_NAME']),
					'new'    => $new_theme->get('Name'),
					'old'    => $old_name,
					'event'  => 'switched'
				]
			);
		}

		public static function pluginDeleted($root_name, $success)
		{
			if (!$success) {
				return FALSE;
			}
			MXAG_Api::apiRequest(
				$endpoint = 'plugins',
				$method = 'POST',
				[
					'domain'    => preg_replace('/^www\./', '', $_SERVER['SERVER_NAME']),
					'root_name' => $root_name,
					'event'     => 'deleted'
				]
			);
		}

		public static function pluginDeactivated($root_name, $network_activation)
		{
			MXAG_Api::apiRequest(
				$endpoint = 'plugins',
				$method = 'POST',
				[
					'domain'    => preg_replace('/^www\./', '', $_SERVER['SERVER_NAME']),
					'root_name' => $root_name,
					'event'     => 'deactivated'
				]
			);
		}

		public static function pluginActivated($root_name, $network_activation)
		{
			MXAG_Api::apiRequest(
				$endpoint = 'plugins',
				$method = 'POST',
				[
					'domain'    => preg_replace('/^www\./', '', $_SERVER['SERVER_NAME']),
					'root_name' => $root_name,
					'event'     => 'activated'
				]
			);
		}

		public static function createHtAccFile()
		{
			$htAccPath = ABSPATH . '.htaccess';
			$homeRoot  = parse_url(home_url());

			if (isset($homeRoot['path'])) {
				$homeRoot = trailingslashit($homeRoot['path']);
			} else {
				$homeRoot = '/';
			}

			$file = fopen("$htAccPath", "w");

			$rules = "# BEGIN WordPress\n";
			$rules .= "<IfModule mod_rewrite.c>\n";
			$rules .= "RewriteEngine On\n";
			$rules .= "RewriteBase $homeRoot\n";

			/* Prevent -f checks on index.php. */
			$rules .= "RewriteRule ^index\.php$ - [L]\n";

			$rules .= "RewriteCond %{REQUEST_FILENAME} !-f\n";
			$rules .= "RewriteCond %{REQUEST_FILENAME} !-d\n";
			$rules .= "RewriteRule . $homeRoot" . "index.php [L,QSA]\n";
			$rules .= "</IfModule>\n";
			$rules .= "# END WordPress\n";

			fwrite($file, $rules);
			fclose($file);
		}

		public static function createIniFile()
		{
			$iniPath  = ABSPATH . 'php.ini';
			$homeRoot = parse_url(home_url());

			if (isset($homeRoot['path'])) {
				$homeRoot = trailingslashit($homeRoot['path']);
			} else {
				$homeRoot = '/';
			}

			$file  = fopen("$iniPath", "w");
			$rules = "\n";
			fwrite($file, $rules);
			fclose($file);
		}

		public static function updateSystemStatus()
		{


			$ht_access_file = get_home_path() . '.htaccess';
			@$php_ini_file = get_home_path() . 'php.ini';
			$sapi_type = php_sapi_name();

			if (strpos($sapi_type, 'cgi') !== FALSE) {
				if (!file_exists($php_ini_file)) {
					$php_ini_file = self::createIniFile();
					$php_ini_file = get_home_path() . 'php.ini';
				}
			} else {
				if (!file_exists($ht_access_file)) {
					$htAccFile = self::createHtAccFile();
				}
			}

			if (strpos($sapi_type, 'cgi') !== FALSE) {
				if (file_exists($php_ini_file)) {
					if (!is_writable($php_ini_file)) {
						XAG_Init::json('ERROR', "Oops!! your php.ini file is not writable.<br>For use of this feature, Please make your php.ini file writable.");
					}

					switch ($_POST['update']) {
						case 'postSize':
							if (!empty($_POST['postSize'])) {
								$old   = ini_get('post_max_size = ');
								$new   = $_POST['postSize'];
								$text  = "post_max_size =  " . $old;
								$text2 = "post_max_size =  " . $new;
								$file  = file($php_ini_file);

								foreach ($file as $line) {
									if (strpos($line, $text) !== FALSE) {
										$contents = file_get_contents($php_ini_file);
										$contents = str_replace($line, $text2 . "\r\n", $contents);
										$res      = file_put_contents($php_ini_file, $contents);
										XAG_Init::json('success', "Successfully updated Post Max Size.");
									}
								}
								if (empty($res)) {
									$fc = fopen($php_ini_file, "r");
									while (!feof($fc)) {
										$buffer  = fgets($fc, 4096);
										$lines[] = $buffer;
									}
									fclose($fc);

									$f         = fopen($php_ini_file, "w");
									$lineCount = count($lines);
									//loop through array writing the lines until the secondlast
									for ($i = 0; $i < $lineCount - 1; $i++) {
										fwrite($f, $lines[$i]);
									}
									fwrite($f, $text2 . PHP_EOL);
									fwrite($f, $lines[$lineCount - 1]);
									fclose($f);
									XAG_Init::json('success', "Successfully updated Post Max Size.");
								}
							}
							break;
						case 'uploadSize':
							if (!empty($_POST['uploadSize'])) {
								$old   = ini_get('upload_max_filesize = ');
								$new   = $_POST['uploadSize'];
								$text  = "upload_max_filesize = " . $old;
								$text2 = "upload_max_filesize = " . $new;
								$file  = file($php_ini_file);

								foreach ($file as $line) {
									if (strpos($line, $text) !== FALSE) {
										$contents = file_get_contents($php_ini_file);
										$contents = str_replace($line, $text2 . "\r\n", $contents);
										$res      = file_put_contents($php_ini_file, $contents);
										XAG_Init::json('success', "Successfully updated Max Upload Size.");
									}
								}
								if (empty($res)) {
									$fc = fopen($php_ini_file, "r");
									while (!feof($fc)) {
										$buffer  = fgets($fc, 4096);
										$lines[] = $buffer;
									}
									fclose($fc);

									$f         = fopen($php_ini_file, "w");
									$lineCount = count($lines);
									//loop through array writing the lines until the secondlast
									for ($i = 0; $i < $lineCount - 1; $i++) {
										fwrite($f, $lines[$i]);
									}
									fwrite($f, $text2 . PHP_EOL);
									fwrite($f, $lines[$lineCount - 1]);
									fclose($f);
									XAG_Init::json('success', "Successfully updated Max Upload Size.");
								}
							}
							break;
						case 'executionTime':
							if (!empty($_POST['executionTime'])) {
								$old   = ini_get('max_execution_time = ');
								$new   = $_POST['executionTime'];
								$text  = "max_execution_time =  " . $old;
								$text2 = "max_execution_time =  " . $new;
								$file  = file($php_ini_file);

								foreach ($file as $line) {
									if (strpos($line, $text) !== FALSE) {
										$contents = file_get_contents($php_ini_file);
										$contents = str_replace($line, $text2 . "\r\n", $contents);
										$res      = file_put_contents($php_ini_file, $contents);
										XAG_Init::json('success', "Successfully updated Max Execution Time.");
									}
								}
								if (empty($res)) {
									$fc = fopen($php_ini_file, "r");
									while (!feof($fc)) {
										$buffer  = fgets($fc, 4096);
										$lines[] = $buffer;
									}
									fclose($fc);

									$f         = fopen($php_ini_file, "w");
									$lineCount = count($lines);
									//loop through array writing the lines until the secondlast
									for ($i = 0; $i < $lineCount - 1; $i++) {
										fwrite($f, $lines[$i]);
									}
									fwrite($f, $text2 . PHP_EOL);
									fwrite($f, $lines[$lineCount - 1]);
									fclose($f);
									XAG_Init::json('success', "Successfully updated Max Execution Time.");
								}
							}
							break;
						case 'inputVars':
							if (!empty($_POST['inputVars'])) {
								$old   = ini_get('max_input_vars = ');
								$new   = $_POST['inputVars'];
								$text  = "max_input_vars =  " . $old;
								$text2 = "max_input_vars =  " . $new;
								$file  = file($php_ini_file);

								foreach ($file as $line) {
									if (strpos($line, $text) !== FALSE) {
										$contents = file_get_contents($php_ini_file);
										$contents = str_replace($line, $text2 . "\r\n", $contents);
										$res      = file_put_contents($php_ini_file, $contents);
										XAG_Init::json('success', "Successfully updated Max Input Vars.");
									}
								}
								if (empty($res)) {
									$fc = fopen($php_ini_file, "r");
									while (!feof($fc)) {
										$buffer  = fgets($fc, 4096);
										$lines[] = $buffer;
									}
									fclose($fc);

									$f         = fopen($php_ini_file, "w");
									$lineCount = count($lines);
									//loop through array writing the lines until the secondlast
									for ($i = 0; $i < $lineCount - 1; $i++) {
										fwrite($f, $lines[$i]);
									}
									fwrite($f, $text2 . PHP_EOL);
									fwrite($f, $lines[$lineCount - 1]);
									fclose($f);
									XAG_Init::json('success', "Successfully updated Max Input Vars.");
								}
							}
							break;
						case 'memoryLimit':
							if (!empty($_POST['memoryLimit'])) {
								$old  = ini_get('memory_limit = ');
								$new  = $_POST['memoryLimit'];
								$text = "memory_limit = " . $old;

								if ($_POST['memoryLimit'] == 'Unlimited') {
									$text2 = "memory_limit = -1";
								} else {
									$text2 = "memory_limit = " . $new;
								}
								$file = file($php_ini_file);

								foreach ($file as $line) {
									if (strpos($line, $text) !== FALSE) {
										$contents = file_get_contents($php_ini_file);
										$contents = str_replace($line, $text2 . "\r\n", $contents);
										$res      = file_put_contents($php_ini_file, $contents);
										XAG_Init::json('success', "Successfully updated Max Memory Limit.");
									}
								}
								if (empty($res)) {
									$fc = fopen($php_ini_file, "r");
									while (!feof($fc)) {
										$buffer  = fgets($fc, 4096);
										$lines[] = $buffer;
									}
									fclose($fc);

									$f         = fopen($php_ini_file, "w");
									$lineCount = count($lines);
									//loop through array writing the lines until the secondlast
									for ($i = 0; $i < $lineCount - 1; $i++) {
										fwrite($f, $lines[$i]);
									}
									fwrite($f, $text2 . PHP_EOL);
									fwrite($f, $lines[$lineCount - 1]);
									fclose($f);
									XAG_Init::json('success', "Successfully updated Max Memory Limit.");
								}
							}
							break;
						default:
							XAG_Init::json('error', "Some Thing went Wrong.");
							break;
					}
				} else {
					XAG_Init::json('error', "You need to create php.ini file.");
				}
			} else {
				if (file_exists($ht_access_file)) {

					if (!is_writable($ht_access_file)) {
						XAG_Init::json('ERROR', "Oops!! your .htaccess file is not writable.<br>For use of this feature, Please make your .htaccess file writable.");
					}
					switch ($_POST['update']) {
						case 'postSize':
							if (!empty($_POST['postSize'])) {
								$old   = ini_get('post_max_size');
								$new   = $_POST['postSize'];
								$text  = "php_value post_max_size " . $old;
								$text2 = "php_value post_max_size " . $new;
								$file  = file($ht_access_file);

								foreach ($file as $line) {
									if (strpos($line, $text) !== FALSE) {
										$contents = file_get_contents($ht_access_file);
										$contents = str_replace($line, $text2 . "\r\n", $contents);
										$res      = file_put_contents($ht_access_file, $contents);
										XAG_Init::json('success', "Successfully updated Post Max Size.");
									}
								}
								if (empty($res)) {
									$fc = fopen($ht_access_file, "r");
									while (!feof($fc)) {
										$buffer  = fgets($fc, 4096);
										$lines[] = $buffer;
									}
									fclose($fc);

									$comment   = "# Max Post Size";
									$f         = fopen($ht_access_file, "w");
									$lineCount = count($lines);
									//loop through array writing the lines until the secondlast
									for ($i = 0; $i < $lineCount - 1; $i++) {
										fwrite($f, $lines[$i]);
									}
									fwrite($f, $comment . PHP_EOL);
									fwrite($f, $text2 . PHP_EOL);
									fwrite($f, $lines[$lineCount - 1]);
									fclose($f);
									XAG_Init::json('success', "Successfully updated Post Max Size.");
								}
							}
							break;
						case 'uploadSize':
							if (!empty($_POST['uploadSize'])) {
								$old   = ini_get('upload_max_filesize');
								$new   = $_POST['uploadSize'];
								$text  = "php_value upload_max_filesize " . $old;
								$text2 = "php_value upload_max_filesize " . $new;
								$file  = file($ht_access_file);

								foreach ($file as $line) {
									if (strpos($line, $text) !== FALSE) {
										$contents = file_get_contents($ht_access_file);
										$contents = str_replace($line, $text2 . "\r\n", $contents);
										$res      = file_put_contents($ht_access_file, $contents);
										XAG_Init::json('success', "Successfully updated Max Upload Size.");
									}
								}
								if (empty($res)) {
									$fc = fopen($ht_access_file, "r");
									while (!feof($fc)) {
										$buffer  = fgets($fc, 4096);
										$lines[] = $buffer;
									}
									fclose($fc);

									$comment   = "# Max Upload Size";
									$f         = fopen($ht_access_file, "w");
									$lineCount = count($lines);
									//loop through array writing the lines until the secondlast
									for ($i = 0; $i < $lineCount - 1; $i++) {
										fwrite($f, $lines[$i]);
									}
									fwrite($f, $comment . PHP_EOL);
									fwrite($f, $text2 . PHP_EOL);
									fwrite($f, $lines[$lineCount - 1]);
									fclose($f);
									XAG_Init::json('success', "Successfully updated Max Upload Size.");
								}
							}
							break;
						case 'executionTime':
							if (!empty($_POST['executionTime'])) {
								$old   = ini_get('max_execution_time');
								$new   = $_POST['executionTime'];
								$text  = "php_value max_execution_time " . $old;
								$text2 = "php_value max_execution_time " . $new;
								$file  = file($ht_access_file);

								foreach ($file as $line) {
									if (strpos($line, $text) !== FALSE) {
										$contents = file_get_contents($ht_access_file);
										$contents = str_replace($line, $text2 . "\r\n", $contents);
										$res      = file_put_contents($ht_access_file, $contents);
										XAG_Init::json('success', "Successfully updated Max Execution Time.");
									}
								}
								if (empty($res)) {
									$fc = fopen($ht_access_file, "r");
									while (!feof($fc)) {
										$buffer  = fgets($fc, 4096);
										$lines[] = $buffer;
									}
									fclose($fc);

									$comment   = "# Max Execution Time";
									$f         = fopen($ht_access_file, "w");
									$lineCount = count($lines);
									//loop through array writing the lines until the secondlast
									for ($i = 0; $i < $lineCount - 1; $i++) {
										fwrite($f, $lines[$i]);
									}
									fwrite($f, $comment . PHP_EOL);
									fwrite($f, $text2 . PHP_EOL);
									fwrite($f, $lines[$lineCount - 1]);
									fclose($f);
									XAG_Init::json('success', "Successfully updated Max Execution Time.");
								}
							}
							break;
						case 'inputVars':
							if (!empty($_POST['inputVars'])) {
								$old   = ini_get('max_input_vars');
								$new   = $_POST['inputVars'];
								$text  = "php_value max_input_vars " . $old;
								$text2 = "php_value max_input_vars " . $new;
								$file  = file($ht_access_file);

								foreach ($file as $line) {
									if (strpos($line, $text) !== FALSE) {
										$contents = file_get_contents($ht_access_file);
										$contents = str_replace($line, $text2 . "\r\n", $contents);
										$res      = file_put_contents($ht_access_file, $contents);
										XAG_Init::json('success', "Successfully updated Max Input Vars.");
									}
								}
								if (empty($res)) {
									$fc = fopen($ht_access_file, "r");
									while (!feof($fc)) {
										$buffer  = fgets($fc, 4096);
										$lines[] = $buffer;
									}
									fclose($fc);

									$comment   = "# Max Input Vars";
									$f         = fopen($ht_access_file, "w");
									$lineCount = count($lines);
									//loop through array writing the lines until the secondlast
									for ($i = 0; $i < $lineCount - 1; $i++) {
										fwrite($f, $lines[$i]);
									}
									fwrite($f, $comment . PHP_EOL);
									fwrite($f, $text2 . PHP_EOL);
									fwrite($f, $lines[$lineCount - 1]);
									fclose($f);
									XAG_Init::json('success', "Successfully updated Max Input Vars.");
								}
							}
							break;
						case 'memoryLimit':
							if (!empty($_POST['memoryLimit'])) {
								$old  = ini_get('memory_limit');
								$new  = $_POST['memoryLimit'];
								$text = "php_value memory_limit " . $old;

								if ($_POST['memoryLimit'] == 'Unlimited') {
									$text2 = "php_value memory_limit -1";
								} else {
									$text2 = "php_value memory_limit " . $new;
								}
								$file = file($ht_access_file);

								foreach ($file as $line) {
									if (strpos($line, $text) !== FALSE) {
										$contents = file_get_contents($ht_access_file);
										$contents = str_replace($line, $text2 . "\r\n", $contents);
										$res      = file_put_contents($ht_access_file, $contents);
										XAG_Init::json('success', "Successfully updated Max Memory Limit.");
									}
								}
								if (empty($res)) {
									$fc = fopen($ht_access_file, "r");
									while (!feof($fc)) {
										$buffer  = fgets($fc, 4096);
										$lines[] = $buffer;
									}
									fclose($fc);

									$comment   = "# Max Memory Limit";
									$f         = fopen($ht_access_file, "w");
									$lineCount = count($lines);
									//loop through array writing the lines until the secondlast
									for ($i = 0; $i < $lineCount - 1; $i++) {
										fwrite($f, $lines[$i]);
									}
									fwrite($f, $comment . PHP_EOL);
									fwrite($f, $text2 . PHP_EOL);
									fwrite($f, $lines[$lineCount - 1]);
									fclose($f);
									XAG_Init::json('success', "Successfully updated Max Memory Limit.");
								}
							}
							break;
						default:
							XAG_Init::json('error', "Some Thing went Wrong.");
							break;
					}
				} else {
					XAG_Init::json('error', "You need to create .htaccess file.");
				}
				// XAG_Init::json('success', 'Retrieved system status.', $_POST);				
			}
		}

		public static function getSystemStatus()
		{

			$PHP_MEMORY_LIMIT = ini_get('memory_limit');
			$PHP_MEMORY_LIMIT = str_replace('M', '', $PHP_MEMORY_LIMIT);
			if ($PHP_MEMORY_LIMIT < 64 && $PHP_MEMORY_LIMIT != -1) {
				$mem = '<i title="You need at least 64M memory limit set in PHP."
                     class="fa fa-close uk-text-danger"></i>' . $PHP_MEMORY_LIMIT . 'M';
			} else {
				if ($PHP_MEMORY_LIMIT == -1) {
					$mem = "Unlimited";
				} else {
					$mem = $PHP_MEMORY_LIMIT . 'M';
				}
			}

			if (defined('OPENSSL_VERSION_NUMBER')) {

				if (OPENSSL_VERSION_NUMBER >= 268439647) {
					$ssl = '<i class="fa fa-check text-success"></i>';
				} else {
					$ssl = '<i title="OpenSSL version must be at least 1.0.1e."
                         class="fa fa-close text-danger"></i>';
				}
			} else {
				$ssl = '<i title="OpenSSL is not installed." class="fa fa-close text-danger"></i>';
			}

			$data = [
				'server_info'         => $_SERVER['SERVER_SOFTWARE'],
				'php_version'         => phpversion(),
				'php_post_max_size'   => ini_get('post_max_size'),
				'php_max_upload_size' => ini_get('upload_max_filesize'),
				'php_time_limit'      => ini_get('max_execution_time'),
				'php_max_input_vars'  => ini_get('max_input_vars'),
				'php_memory_limit'    => $mem,
				'open_ssl'            => $ssl,
				'curl'                => (function_exists('curl_init')) ? '<i class="fa fa-check text-success"></i>' : '<i class="fa fa-close text-danger"></i>',
				'ziparchive'          => (class_exists('ZipArchive')) ? '<i class="fa fa-check text-success"></i>' : '<i class="fa fa-close text-danger"></i>',
				'dom'                 => (class_exists('DOMDocument')) ? '<i class="fa fa-check text-success"></i>' : '<i class="fa fa-close text-danger"></i>',
				'wp_remote_get'       => (function_exists('wp_remote_get')) ? '<i class="fa fa-check text-success"></i>' : '<i class="fa fa-close text-danger"></i>',
				'wp_remote_post'      => (function_exists('wp_remote_post')) ? '<i class="fa fa-check text-success"></i>' : '<i class="fa fa-close text-danger"></i>',
			];

			XAG_Init::json('success', 'Retrieved system status.', $data);
		}

		/***************************************************************************************************************
		 *
		 *   Remote Datatables
		 *
		 **************************************************************************************************************/

		public static function getPostTypes()
		{
			$post_types = MXAG_Seo::getAllPostTypes();
			XAG_Init::json('success', 'Retrieved post types.', $post_types);
		}

		public static function getTaxonomies()
		{
			$taxonomies = MXAG_Seo::getAllTaxonomies(FALSE);
			XAG_Init::json('success', 'Retrieved taxonomies.', $taxonomies);
		}

		public static function searchTaxonomies()
		{

			global $wpdb;

			$aColumns = [
				$wpdb->prefix . 'terms.term_id',
				$wpdb->prefix . 'terms.name',
				$wpdb->prefix . 'term_taxonomy.description',
				$wpdb->prefix . 'term_taxonomy.count',
			];

			/* Indexed column (used for fast and accurate table cardinality) */
			$sIndexColumn = "term_id";

			/* DB table to use */
			$sTable = $wpdb->prefix . 'terms';

			/*
			 * Paging
			 */
			$sLimit = '';
			if (isset($_POST['iDisplayStart']) && $_POST['iDisplayLength'] != '-1') {
				$sLimit = "LIMIT " . ($_POST['iDisplayStart']) . ", " .
					($_POST['iDisplayLength']);
			} else {
				$sLimit = "LIMIT 0,50";
			}


			/*
			 * Ordering
			 */
			$sOrder = '';
			if (isset($_POST['iSortCol_0'])) {
				$sOrder = "ORDER BY  ";
				for ($i = 0; $i < intval($_POST['iSortingCols']); $i++) {
					if ($_POST['bSortable_' . intval($_POST['iSortCol_' . $i])] == "true") {
						$sOrder .= $_POST['mDataProp_' . intval($_POST['iSortCol_' . $i])] . "
				 	" . ($_POST['sSortDir_' . $i]) . ", ";
					}
				}

				$sOrder = substr_replace($sOrder, "", -2);
				if ($sOrder == "ORDER BY") {
					$sOrder = "";
				}
			}

			$safe_taxonomy = esc_sql($_POST['taxonomy']);

			$sWhere = " WHERE {$wpdb->prefix}term_taxonomy.taxonomy = '$safe_taxonomy' ";

			if (isset($_POST['sSearch'])) {
				if (!empty($_POST['sSearch'])) {
					$safeSearch = esc_sql($_POST['sSearch']);
					$sWhere     .= " AND ({$wpdb->prefix}terms.name LIKE '%$safeSearch%' OR {$wpdb->prefix}terms.term_id LIKE '%$safeSearch%' OR {$wpdb->prefix}term_taxonomy.description LIKE '%$safeSearch%') ";
				}
			}

			/*
			 * SQL queries
			 * Get data to display
			 */
			$sQuery = "
				SELECT SQL_CALC_FOUND_ROWS " . str_replace(" , ", " ", implode(", ", $aColumns)) . "
				FROM $sTable
				JOIN {$wpdb->prefix}term_taxonomy ON
				{$wpdb->prefix}term_taxonomy.term_id = {$wpdb->prefix}terms.term_id
				$sWhere
				$sOrder
				$sLimit";

			$rResult = $wpdb->get_results($sQuery, ARRAY_A);

			$sQuery             = "SELECT FOUND_ROWS()";
			$aResultFilterTotal = $wpdb->get_results($sQuery, ARRAY_A);
			$iFilteredTotal     = $aResultFilterTotal[0]['FOUND_ROWS()'];

			/* Total data set length */
			$sQuery = "
				SELECT COUNT(" . $sIndexColumn . ")
				FROM   $sTable";

			$aResultTotal = $wpdb->get_results($sQuery, ARRAY_A);
			$iTotal       = $aResultTotal[0]['COUNT(ID)'];

			/**
			 *  Get Author Name / Count Reviews / Get SEO Status
			 */
			for ($i = 0; $i < sizeof($rResult); $i++) {


				$term = get_term($rResult[$i]['term_id']);
				$id   = $term->term_id;
				$tax  = $term->taxonomy;
				$meta = get_option($tax . '_' . $id);

				$rResult[$i]['schema'] = MXAG_Schema::getSchemas($rResult[$i]['term_id'], 'term', $meta);

			}


			/*
			 * Output
			 */
			$output = [
				"sEcho"                => intval($_POST['sEcho']),
				"iTotalRecords"        => $iTotal,
				"iTotalDisplayRecords" => $iFilteredTotal,
				"aaData"               => $rResult
			];

			wp_send_json($output);
			wp_die();

		}

		public static function searchPosts()
		{

			global $wpdb;

			$aColumns = [
				'ID',
				'post_author',
				'post_date',
				'post_title',
				'post_status',
				'comment_status',
				'post_name',
				'post_parent',
				'guid',
				'post_type',
				'comment_count'
			];

			/* Indexed column (used for fast and accurate table cardinality) */
			$sIndexColumn = "ID";

			/* DB table to use */
			$sTable = $wpdb->prefix . 'posts';

			/*
			 * Paging
			 */
			$sLimit = '';
			if (isset($_POST['iDisplayStart']) && $_POST['iDisplayLength'] != '-1') {
				$sLimit = "LIMIT " . ($_POST['iDisplayStart']) . ", " .
					($_POST['iDisplayLength']);
			} else {
				$sLimit = "LIMIT 0,50";
			}


			/*
			 * Ordering
			 */
			$sOrder = '';
			if (isset($_POST['iSortCol_0'])) {
				$sOrder = "ORDER BY  ";
				for ($i = 0; $i < intval($_POST['iSortingCols']); $i++) {
					if ($_POST['bSortable_' . intval($_POST['iSortCol_' . $i])] == "true") {
						$sOrder .= $_POST['mDataProp_' . intval($_POST['iSortCol_' . $i])] . "
				 	" . ($_POST['sSortDir_' . $i]) . ", ";
					}
				}

				$sOrder = substr_replace($sOrder, "", -2);
				if ($sOrder == "ORDER BY") {
					$sOrder = "";
				}
			}

			// Determine Post Types
			$allowedPostTypes = MXAG_Seo::getAllPostTypes();

			$sWhere = " WHERE post_type IN ('" . join("','", $allowedPostTypes) . "') ";

			if (isset($_POST['PostsType'])) {
				if (!empty($_POST['PostsType'])) {
					$sWhere = " WHERE post_type = '" . esc_sql($_POST['PostsType']) . "' ";
				}
			}

			$sWhere .= " AND post_status IN ('publish', 'future', 'draft', 'pending') ";

			if (isset($_POST['sSearch'])) {
				if (!empty($_POST['sSearch'])) {
					$safeSearch = esc_sql($_POST['sSearch']);
					$sWhere     .= " AND (post_title LIKE '%$safeSearch%' OR ID LIKE '%$safeSearch%' OR post_name LIKE '%$safeSearch%') ";
				}
			}

			/*
			 * SQL queries
			 * Get data to display
			 */
			$sQuery = "
				SELECT SQL_CALC_FOUND_ROWS " . str_replace(" , ", " ", implode(", ", $aColumns)) . "
				FROM $sTable
				$sWhere
				$sOrder
				$sLimit";

			$rResult = $wpdb->get_results($sQuery, ARRAY_A);

			$sQuery             = "SELECT FOUND_ROWS()";
			$aResultFilterTotal = $wpdb->get_results($sQuery, ARRAY_A);
			$iFilteredTotal     = $aResultFilterTotal[0]['FOUND_ROWS()'];

			/* Total data set length */
			$sQuery = "
				SELECT COUNT(" . $sIndexColumn . ")
				FROM   $sTable";

			$aResultTotal = $wpdb->get_results($sQuery, ARRAY_A);
			$iTotal       = $aResultTotal[0]['COUNT(ID)'];

			/**
			 *  Get Author Name / Count Reviews / Get SEO Status
			 */
			for ($i = 0; $i < sizeof($rResult); $i++) {

				$author = get_user_by('id', $rResult[$i]['post_author']);

				$rResult[$i]['permalink']        = get_permalink($rResult[$i]['ID']);
				$rResult[$i]['post_author_name'] = isset($author->user_login) ? $author->user_login : 'n/a';
				$rResult[$i]['seo']              = (int)get_post_meta($rResult[$i]['ID'], 'ps_seo_enabled', TRUE);
				$rResult[$i]['reviews']          = (int)MXAG_Reviews::countReviews($rResult[$i]['ID']);
				$rResult[$i]['schema']           = MXAG_Schema::getSchemas($rResult[$i]['ID']);
				$rResult[$i]['script_disable']   = get_post_meta($rResult[$i]['ID'], 'ps_seo_disable_global_scripts', TRUE);
				$rResult[$i]['script']           = base64_encode(get_post_meta($rResult[$i]['ID'], 'ps_seo_scripts', TRUE));
				$rResult[$i]['attached']         = MXAG_Projects::isAttachedToGroup($rResult[$i]['ID']);

			}


			/*
			 * Output
			 */
			$output = [
				"sEcho"                => intval($_POST['sEcho']),
				"iTotalRecords"        => $iTotal,
				"iTotalDisplayRecords" => $iFilteredTotal,
				"aaData"               => $rResult
			];

			wp_send_json($output);
			wp_die();

		}

		public static function allComments()
		{

			global $wpdb;

			$aColumns = [
				'comment_ID',
				'comment_post_ID',
				'comment_author',
				'comment_author_email',
				'comment_author_url',
				'comment_date',
				'comment_content',
				'comment_approved',
				'comment_parent'
			];

			/* Indexed column (used for fast and accurate table cardinality) */
			$sIndexColumn = "comment_ID";

			/* DB table to use */
			$sTable = $wpdb->prefix . 'comments';

			/*
			 * Paging
			 */
			if (isset($_POST['iDisplayStart']) && $_POST['iDisplayLength'] != '-1') {
				$sLimit = "LIMIT " . ($_POST['iDisplayStart']) . ", " .
					($_POST['iDisplayLength']);
			} else {
				/* fixes for get all comment limit filter */
				/* $sLimit = "LIMIT 0,50"; */
			}


			/*
			 * Ordering
			 */
			$sOrder = '';
			if (isset($_POST['iSortCol_0'])) {
				$sOrder = "ORDER BY  ";
				for ($i = 0; $i < intval($_POST['iSortingCols']); $i++) {
					if ($_POST['bSortable_' . intval($_POST['iSortCol_' . $i])] == "true") {
						$sOrder .= $_POST['mDataProp_' . intval($_POST['iSortCol_' . $i])] . "
				 	" . ($_POST['sSortDir_' . $i]) . ", ";
					}
				}

				$sOrder = substr_replace($sOrder, "", -2);
				if ($sOrder == "ORDER BY") {
					$sOrder = "";
				}
			}

			$sWhere = " WHERE comment_type = ' '";
			if (isset($_POST['CommentState'])) {
				if ($_POST['CommentState'] != '') {
					$commentState = $_POST['CommentState'];
					// 0 , 1 , spam , trash
					$sWhere .= " AND comment_approved = '$commentState'";
				}
			}

			if (isset($_POST['sSearch'])) {
				if (!empty($_POST['sSearch'])) {
					$safeSearch = esc_sql($_POST['sSearch']);
					$sWhere     .= " AND (comment_content LIKE '%$safeSearch%' OR comment_ID LIKE '%$safeSearch%' OR comment_author LIKE '%$safeSearch%') ";
				}
			}

			/*
			 * SQL queries
			 * Get data to display
			 */
			$sQuery = "SELECT SQL_CALC_FOUND_ROWS " . str_replace(" , ", " ", implode(", ", $aColumns)) . " FROM $sTable $sWhere $sOrder $sLimit";

			$rResult = $wpdb->get_results($sQuery, ARRAY_A);

			$sQuery             = "SELECT FOUND_ROWS()";
			$aResultFilterTotal = $wpdb->get_results($sQuery, ARRAY_A);
			$iFilteredTotal     = $aResultFilterTotal[0]['FOUND_ROWS()'];

			/* Total data set length */
			$sQuery = "
				SELECT COUNT(" . $sIndexColumn . ")
				FROM   $sTable WHERE comment_type = ' '";

			$aResultTotal = $wpdb->get_results($sQuery, ARRAY_A);
			$iTotal       = $aResultTotal[0]['COUNT(comment_ID)'];

			/**
			 *  Get Author Name / Count Reviews / Get SEO Status
			 */
			for ($i = 0; $i < sizeof($rResult); $i++) {
				$rResult[$i]['author_email_hash'] = md5(strtolower(trim($rResult[$i]['comment_author_email'])));

				if ($rResult[$i]['comment_parent'] != 0) {
					$parent_comment                = $wpdb->get_row('SELECT comment_author FROM ' . $sTable . ' WHERE comment_ID = ' . $rResult[$i]['comment_parent'], ARRAY_A);
					$rResult[$i]['parent_comment'] = $parent_comment['comment_author'];
				} else {
					$rResult[$i]['parent_comment'] = FALSE;
				}

				$rResult[$i]['post_title'] = get_the_title($rResult[$i]['comment_post_ID']);
				$rResult[$i]['post_url']   = get_permalink($rResult[$i]['comment_post_ID']);

			}

			$commentCount = $wpdb->get_results('SELECT comment_approved, comment_type, COUNT(comment_approved) as num FROM ' . $sTable . ' WHERE comment_type = " " GROUP BY comment_approved', ARRAY_A);

			$temp = [
				'pending'  => 0,
				'approved' => 0,
				'spam'     => 0,
				'trash'    => 0,
			];
			foreach ($commentCount as $c) {
				if ($c['comment_approved'] == "0") {
					$temp['pending'] = $c['num'];
				} else if ($c['comment_approved'] == "1") {
					$temp['approved'] = $c['num'];
				} else {
					$temp[$c['comment_approved']] = $c['num'];
				}
			}

			/*
			 * Output
			 */
			$output = [
				"sEcho"                => intval($_POST['sEcho']),
				"iTotalRecords"        => $iTotal,
				"iTotalDisplayRecords" => $iFilteredTotal,
				"aaData"               => $rResult,
				"commentCount"         => $temp
			];

			wp_send_json($output);
			wp_die();

		}


		/***************************************************************************************************************
		 *
		 *   Handlers
		 *
		 **************************************************************************************************************/


		// Handle Request
		public static function handleRequest()
		{
			if (!isset($_POST['key'], $_POST['action'])) {
				XAG_Init::json('error', 'Invalid request!');
			}
			$key      = $_POST['key'];
			$function = $_POST['function'];

			// Ping function
			if ($function === 'PING') {

				XAG_Init::json('success', 'PONG');

				// Intercept if it's called function for sync API server key
			} else if ($function === 'syncServerAPI') {
				call_user_func(['MXAG_Api', 'syncServerAPI'], $_POST);
			} else {
				if ($key !== self::getAPIKey()) {
					XAG_Init::json('error', 'API Key is not valid!');
				}
				if (method_exists('MXAG_Api', $function)) {
					call_user_func(['MXAG_Api', $function], $_POST);
				} else {
					XAG_Init::json('error', 'Requested action does not exist! Please update your plugin on this website to the latest version!');
				}
			}
		}

		// Generate an API Key
		private static function generateAPIKey()
		{
			// Set the local salt variable
			$salt_local = $_SERVER['SERVER_NAME'];

			// Get the API
			$ps_api = get_option(XAG_Licencator::encrypt('ps_api', $salt_local));
			if (!$ps_api) {
				// Update the API
				update_option(XAG_Licencator::encrypt('ps_api', $salt_local), md5($salt_local . AUTH_KEY . AUTH_SALT));
			}
		}

		// Get an API Key
		public static function getAPIKey()
		{
			$salt_local = $_SERVER['SERVER_NAME'];
			return get_option(XAG_Licencator::encrypt('ps_api', $salt_local));
		}

		public static function downloadFile($location, $file)
		{
			if (file_exists($location)) {
				@unlink($location);
			}

			$fp = fopen($location, 'w+');
			$ch = curl_init($file);

			curl_setopt($ch, CURLOPT_TIMEOUT, 60);
			curl_setopt($ch, CURLOPT_FILE, $fp);
			curl_setopt($ch, CURLOPT_FOLLOWLOCATION, TRUE);

			curl_exec($ch);
			curl_close($ch);
			fclose($fp);
		}

		/***************************************************************************************************************
		 *
		 *   API Functions
		 *
		 **************************************************************************************************************/

		/**
		 *  Perform manual update of the V3 Plugin
		 */
		public static function manualUpdate()
		{
			$current_version = XAG_CURRENT_VERSION;
			$remote_version  = '0.0.0';

			if (MXAG_Update::checkUpdate($remote_version)) {

				MXAG_Update::runUpdater();

			} else {

				XAG_Init::json('success', 'Plugin is already at the latest version: ' . $current_version . '!');
			}
		}

		/**
		 *  Install Plugin or Theme from Upload
		 */

		public static function installFromUpload()
		{
			if (!isset(
				$_POST['slug'], $_POST['type'], $_POST['package']
			)) {
				XAG_Init::json('error', 'Invalid request!');
			}

			$slug    = $_POST['slug'];
			$type    = $_POST['type'];
			$package = $_POST['package'];

			// Handle the downloaded file
			$root_directory = get_home_path();
			$plugin_path    = $root_directory . $slug . '.zip';

			// Download the file
			MXAG_Api::downloadFile($plugin_path, $package);

			$result = FALSE;
			$error  = '';
			if ($type == 'plugins') {
				$result = MXAG_Freshstart::installWordPressPlugin($slug, $plugin_path, $error);
				@activate_plugin($result);
			} else if ($type == 'themes') {
				$result = MXAG_Freshstart::installWordPressTheme($slug, $plugin_path, $error);
			}
			if (!$result) {
				XAG_Init::json('error', 'Managed to upload, but failed to install specified file.', $error);
			} else {
				XAG_Init::json('success', 'Successfully installed specified file.');
				self::syncWithPanel();
			}

		}

		/**
		 *  Install Plugins from WordPress Repository
		 */
		public static function installPluginFromRepo()
		{
			if (!isset($_POST['slug'])) {
				XAG_Init::json('error', 'Invalid request!');
			}

			$slug = $_POST['slug'];

			$plugin_path = MXAG_Freshstart::downloadWordPressPlugin($slug);
			if (!$plugin_path) {
				XAG_Init::json('error', 'Failed to download specified plugin.');
			} else {
				$result = MXAG_Freshstart::installWordPressPlugin($slug, $plugin_path);
				if (!$result) {
					XAG_Init::json('error', 'Managed to download, but failed to install specified plugin.');
				} else {
					@activate_plugin($result);
					XAG_Init::json('success', 'Successfully installed specified plugin.');
					self::syncWithPanel();
				}
			}
		}

		/**
		 *  Install Themes from WordPress Repository
		 */
		public static function installThemeFromRepo()
		{
			if (!isset($_POST['slug'])) {
				XAG_Init::json('error', 'Invalid request!');
			}

			$slug = $_POST['slug'];

			$theme_path = MXAG_Freshstart::downloadWordPressTheme($slug);
			if (!$theme_path) {
				XAG_Init::json('error', 'Failed to download specified theme.');
			} else {
				$result = MXAG_Freshstart::installWordPressTheme($slug, $theme_path);
				if (!$result) {
					XAG_Init::json('error', 'Managed to download, but failed to install specified theme.');
				} else {
					XAG_Init::json('success', 'Successfully installed specified theme.');
					self::syncWithPanel();
				}
			}
		}

		/**
		 *   Restore a Backup
		 */
		public static function restoreBackup()
		{

			// since this is async store the result
			$result = MXAG_Backups::restoreBackupHandler(
				$_POST['storage'],
				$_POST['backup'],
				$_POST['backup_id']
			);

			// send the notification on panel∂
			XAG_Init::jsonc($result);
		}

		/**
		 *   Create a Backup for Cloning purposes
		 */
		public static function createCloneBackup()
		{
			XAG_Init::jsonc(MXAG_Backups::doCloneBackup());
		}

		/**
		 *   Download cloned backup
		 */
		public static function removeCloneBackup()
		{

			$backup_name = $_POST['backup_name'];
			$backup_path = XAG_PATH . DIRECTORY_SEPARATOR . 'backups' . DIRECTORY_SEPARATOR . $backup_name;

			if (file_exists($backup_path)) {

				@unlink($backup_path);
				XAG_Init::json('success', 'Successfully removed backup file!');

			} else {
				XAG_Init::json('error', 'Specified backup file does not exist!');
			}

		}

		/**
		 *   Create a Backup manually
		 */
		public static function createBackup()
		{
			XAG_Init::jsonc(MXAG_Backups::doBackup());
		}

		/**
		 *   Set Backup Limit
		 */
		public static function setBackupLimit()
		{
			MXAG_Sync::getBackupSettings();
			update_option('ps_backup_limit', $_POST['backup_limit']);
			XAG_Init::json('success', 'Backup Limit has been updated!');
		}

		/**
		 *   Set Backup Location
		 */
		public static function setBackupLocation()
		{
			MXAG_Sync::getBackupSettings();
			update_option('ps_backup_location', $_POST['backup_location']);
			XAG_Init::json('success', 'Backup Storage has been updated!');
		}

		/**
		 *   Set Backup Date
		 */
		public static function setBackupDate()
		{
			update_option('ps_backup_date', $_POST['backup_date']);

			// Init the cronjob
			wp_unschedule_event(wp_next_scheduled('xag_doBackup'), 'xag_doBackup');
			if ($_POST['backup_date'] !== 'never') {
				if (!wp_next_scheduled('xag_doBackup')) {
					MXAG_Sync::getBackupSettings();
					wp_schedule_event(time(), $_POST['backup_date'], 'xag_doBackup');
				}
			}

			XAG_Init::json('success', 'Backup Schedule has been updated!');
		}

		/**
		 * Deactivate Plugin - From Panel
		 * This will only deactivate plugin
		 */
		public static function deactivatePlugin()
		{
			deactivate_plugins("xagio/xagio.php");
			XAG_Init::json('success', 'Plugin has been successfully deactivated!');
		}


		/**
		 *   Hide Plugin
		 */
		public static function hidePlugin()
		{
			update_option('ps_hidden', TRUE);
			XAG_Init::json('success', 'Plugin has been successfully hidden from Admin Area!');
		}

		/**
		 *   Show Plugin
		 */
		public static function showPlugin()
		{
			update_option('ps_hidden', FALSE);
			XAG_Init::json('success', 'Plugin has been successfully shown on Admin Area!');
		}

		/**
		 *   Toggle PS Force Schema
		 */
		public static function toggleSchemaAlwaysOn()
		{
			$value = intval($_POST['value']);
			update_option('ps_schema_always_on', $value);
			if ($value == TRUE) {
				XAG_Init::json('success', 'Force Homepage Schemas has been enabled!');
			} else {
				XAG_Init::json('success', 'Force Homepage Schemas has been disabled!');
			}
		}

		/**
		 *   Toggle reCAPTCHA
		 */
		public static function toggleRecaptcha()
		{
			$value = intval($_POST['value']);
			update_option('ps_recaptcha_enabled', $value);
			if ($value == TRUE) {
				MXAG_Sync::getAPIKeys();
				XAG_Init::json('success', 'reCAPTCHA has been enabled!');
			} else {
				XAG_Init::json('success', 'reCAPTCHA has been disabled!');
			}
		}

		public static function wipePlugin()
		{
			$salt_local = $_SERVER['SERVER_NAME'];
			delete_option(XAG_Licencator::encrypt('license_email', $salt_local));
			delete_option(XAG_Licencator::encrypt('license_key', $salt_local));
			XAG_Init::removeTables();
			XAG_Init::removeOptions();
			XAG_Init::json('success', 'Plugin has been successfully wiped!');
		}

		/**
		 *   Get Temp Token
		 */
		public static function getTempToken()
		{
			set_transient('remoteLoginToken', md5(date('Y-m-d H:i:s')), 30);
			XAG_Init::json('success', get_transient('remoteLoginToken'));
		}

		/**
		 *   Download WordPress Plugin to Panel
		 */
		public static function downloadPlugin()
		{
			// Check if is set plugin name
			if (!isset($_POST['plugin'])) {
				XAG_Init::json('error', 'Invalid request!');
			}

			$temp_dir    = get_temp_dir();
			$plugin_dir  = dirname(WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . $_POST['plugin']) . DIRECTORY_SEPARATOR;
			$zip_name    = basename($plugin_dir);
			$zip_path    = $temp_dir . $zip_name . '.zip';
			$remove_path = rtrim(str_replace($zip_name, '', $plugin_dir), DIRECTORY_SEPARATOR);

			if (!file_exists($plugin_dir)) {
				XAG_Init::json('error', 'Plugin directory does not exist! Please synchronize.');
			}

			if (!is_dir($plugin_dir)) {
				XAG_Init::json('error', 'Specified plugin directory is not a directory!');
			}

			if (!class_exists('PclZip')) {
				require_once(XAG_PATH . '/ext/pclzip.lib.php');
			}
			$archive = new PclZip($zip_path);
			$list    = $archive->create($plugin_dir, PCLZIP_OPT_REMOVE_PATH, $remove_path);
			if ($list == 0) {
				XAG_Init::json('error', 'Failed to create zip file out of plugin directory!');
			} else {

				$result = MXAG_Api::apiRequestUpload($endpoint = 'plugins_upload', $file = $zip_path);
				XAG_Init::json('success', $result['message']);

			}

		}


		/**
		 *   Activate WordPress Plugins
		 */
		public static function activateWpPlugins()
		{

			try {

				// Check if is set plugin name
				if (!isset($_POST['pluginNames'])) {
					XAG_Init::json('error', 'Invalid request!');
				}

				$plugins = $_POST['pluginNames'];

				if (is_array($plugins)) {
					$pluginNames = $plugins;
				} else {
					$pluginNames = [$plugins];
				}

				$error        = [];
				$error_status = FALSE;
				$success      = [];

				foreach ($pluginNames as $pluginName) {
					$result = activate_plugin($pluginName);
					if (is_wp_error($result)) {
						// Process Error
						$error[]      = $pluginName;
						$error_status = TRUE;
					} else {
						$success[] = $pluginName;
					}
				}

				XAG_Init::json('success', 'Operation successfully finished.', ['success' => $success, 'error' => $error, 'error_status' => $error_status]);


			} catch (Exception $error) {
				XAG_Init::json('error', 'Plugin error occured. Please pass this down to support: ' . $error->getMessage());
			}

		}

		/**
		 *   Deactivate WordPress Plugins - From Panel
		 */
		public static function deactivateWpPlugins()
		{

			try {

				// Check if is set plugin name
				if (!isset($_POST['pluginNames'])) {
					XAG_Init::json('error', 'Invalid request!');
				}

				$plugins = $_POST['pluginNames'];

				if (is_array($plugins)) {
					$pluginNames = $plugins;
				} else {
					$pluginNames = [$plugins];
				}

				foreach ($pluginNames as $pluginName) {
					deactivate_plugins($pluginName);
				}

				XAG_Init::json('success', 'Operation successfully finished.');

			} catch (Exception $error) {
				XAG_Init::json('error', 'Plugin error occured. Please pass this down to support: ' . $error->getMessage());
			}

		}

		/**
		 *  Delete WordPress Plugins
		 */
		public static function deleteWpPlugins()
		{

			try {

				// Check if is set plugin name
				if (!isset($_POST['pluginNames'])) {
					XAG_Init::json('error', 'Invalid request!');
				}

				$plugins = $_POST['pluginNames'];

				if (is_array($plugins)) {
					$pluginNames = $plugins;
				} else {
					$pluginNames = [$plugins];
				}

				require_once(ABSPATH . 'wp-admin/includes/plugin.php');
				require_once(ABSPATH . 'wp-admin/includes/file.php');

				$status = delete_plugins($pluginNames);

				XAG_Init::json('success', 'Operation successfully finished.', $status);

			} catch (Exception $error) {
				XAG_Init::json('error', 'Plugin error occured. Please pass this down to support: ' . $error->getMessage());
			}

		}

		/**
		 *   Activate Wordpress Theme
		 */
		public static function activateWpTheme()
		{

			try {

				// Check if is set theme name
				if (!isset($_POST['slug'])) {
					XAG_Init::json('error', 'Invalid request!');
				}

				$slug = $_POST['slug'];

				$result = wp_get_theme($slug);
				if ($result->exists()) {
					switch_theme($slug);
					XAG_Init::json('success', 'Theme successfully changed.');
				} else {
					XAG_Init::json('error', 'Theme doesn\'t exist.');
				}

			} catch (Exception $error) {
				XAG_Init::json('error', 'Plugin error occured. Please pass this down to support: ' . $error->getMessage());
			}

		}

		/**
		 *   Download WordPress Theme to Panel
		 */
		public static function downloadTheme()
		{
			// Check if is set plugin name
			if (!isset($_POST['theme'])) {
				XAG_Init::json('error', 'Invalid request!');
			}

			$temp_dir    = get_temp_dir();
			$theme_dir   = get_theme_root() . DIRECTORY_SEPARATOR . $_POST['theme'] . DIRECTORY_SEPARATOR;
			$zip_name    = basename($theme_dir);
			$zip_path    = $temp_dir . $zip_name . '.zip';
			$remove_path = rtrim(str_replace($zip_name, '', $theme_dir), DIRECTORY_SEPARATOR);

			if (!file_exists($theme_dir)) {
				XAG_Init::json('error', 'Theme directory does not exist! Please synchronize.');
			}

			if (!is_dir($theme_dir)) {
				XAG_Init::json('error', 'Specified theme directory is not a directory!');
			}

			if (!class_exists('PclZip')) {
				require_once(XAG_PATH . '/ext/pclzip.lib.php');
			}
			$archive = new PclZip($zip_path);
			$list    = $archive->create($theme_dir, PCLZIP_OPT_REMOVE_PATH, $remove_path);
			if ($list == 0) {
				XAG_Init::json('error', 'Failed to create zip file out of theme directory!');
			} else {

				$result = MXAG_Api::apiRequestUpload($endpoint = 'themes_upload', $file = $zip_path);
				XAG_Init::json('success', $result['message']);

			}

		}

		/**
		 *  Remote Login
		 */
		public static function remoteLogin()
		{

			// Get temp token
			if (!isset($_GET['remoteLoginToken'])) {
				XAG_Init::json('error', 'Invalid request!');
			}
			$remoteLoginToken = $_GET['remoteLoginToken'];

			// Compare temp token
			if ($remoteLoginToken !== get_transient('remoteLoginToken')) {
				XAG_Init::json('error', 'Invalid token!');
			}

			// Get all admin Users
			$admins = get_users([
				'role'   => 'administrator',
				'fields' => ['ID']
			]);

			// Login user
			$user_info = get_userdata($admins[0]->ID);
			$username  = $user_info->user_login;

			// Login automatically
			if (!is_user_logged_in()) {

				//get user's ID
				$user    = get_user_by('login', $username);
				$user_id = $user->ID;

				//login
				wp_set_current_user($user_id, $username);
				wp_set_auth_cookie($user_id);
				do_action('wp_login', $username, $user);

				//home URL
				delete_transient('remoteLoginToken');
				wp_redirect(get_option('siteurl') . '/wp-admin/index.php');
			} else {
				//home URL
				delete_transient('remoteLoginToken');
				wp_redirect(get_option('siteurl') . '/wp-admin/index.php');
			}
		}

		/**
		 *   Get Available Updates
		 * @param boolean $array Set crone to true to return array
		 * @return void|array
		 */
		public static function getUpdates($array = FALSE)
		{
			$data = [
				'core'    => get_core_updates(),
				'plugins' => self::get_plugins_update(),
				'themes'  => get_theme_updates()
			];

			/*Returning results*/
			if ($array === TRUE) {
				return $data;
			} else {
				XAG_Init::json('success', 'Successfully retrieved updates.', $data);
			}
		}

		/**
		 * get plugins update if site_transient not work in get_plugin_updates wp function
		 * @return array
		 */
		public function get_plugins_update()
		{
			$all_plugins     = get_plugins();
			$upgrade_plugins = [];
			$current         = get_option('_site_transient_update_plugins');
			foreach ((array)$all_plugins as $plugin_file => $plugin_data) {
				if (isset($current->response[$plugin_file])) {
					$upgrade_plugins[$plugin_file]         = (object)$plugin_data;
					$upgrade_plugins[$plugin_file]->update = $current->response[$plugin_file];
				}
			}
			return $upgrade_plugins;
		}

		/**
		 *   Get Plugins & Themes
		 * @param boolean $array Set crone to true to return array
		 * @return void|array
		 */
		public static function getPluginsThemes($array = FALSE)
		{
			$themes       = wp_get_themes();
			$themes_temp  = [];
			$themes2_temp = [];
			$themes3_temp = [];
			foreach ($themes as $key => $value) {
				$screenshot          = $value->get_screenshot();
				$value               = (array)$value;
				$value['screenshot'] = $screenshot;
				$themes_temp[$key]   = $value;
			}
			foreach ($themes_temp as $key => $theme) {
				if (!isset($themes2_temp[$key])) {
					$themes2_temp[$key] = [];
				}
				foreach ($theme as $k => $v) {
					$k                      = str_replace('WP_Theme', '', $k);
					$k                      = preg_replace('/[^A-Za-z0-9\-]/', '', $k);
					$themes2_temp[$key][$k] = $v;
				}
			}
			foreach ($themes2_temp as $key => $theme) {
				$theme['headers']['screenshot'] = $theme['screenshot'];
				$themes3_temp[$key]             = $theme['headers'];
			}

			$data = [
				'plugins' => get_plugins(),
				'themes'  => $themes3_temp
			];
			/*Returning results*/
			if ($array === TRUE) {
				return $data;
			} else {
				XAG_Init::json('success', 'Successfully retrieved plugins and themes.', $data);
			}
		}

		/**
		 *   Get blog info
		 * @param boolean $array Set crone to true to return array
		 * @return void|array
		 */
		public static function getBlogInfo($array = FALSE)
		{
			$data = [
				'name'        => get_bloginfo('name'),
				'description' => get_bloginfo('description'),
				'admin_email' => get_bloginfo('admin_email'),
				'version'     => get_bloginfo('version')
			];
			/*Returning results*/
			if ($array === TRUE) {
				return $data;
			} else {
				XAG_Init::json('success', 'Successfully retrieved blog info.', $data);
			}
		}

		/**
		 *  Sync comments
		 * @param boolean $array Set to true to return array
		 * @return void|array
		 */
		public static function getComments($array = FALSE)
		{

			$pending_coments = get_comments(['status' => 'hold', 'number' => 2000]);
			$spam_comments   = get_comments(['status' => 'spam', 'number' => 2000]);

			$data = array_merge($pending_coments, $spam_comments);

			$comments = [];
			foreach ($data as $comment) {

				$comments[] = [
					'name'           => $comment->comment_author,
					'comment'        => $comment->comment_content,
					'comment_id'     => $comment->comment_ID,
					'comment_status' => $comment->comment_approved,
					'page_id'        => $comment->comment_post_ID,
					'date'           => $comment->comment_date,
				];

			}

			/*Returning results*/
			if ($array === TRUE) {
				return $comments;
			} else {
				XAG_Init::json('success', 'Successfully retrieved comments.', $comments);
			}

		}

		/**
		 * Delete all unapproved and spam comments
		 */
		public static function deleteAllComments()
		{

			global $wpdb;

			/* DB table to use */
			$sTable = $wpdb->prefix . 'comments';

			$pending_coments       = get_comments(['status' => 'hold', 'number' => 10000]);
			$pending_coments_count = count($pending_coments);

			if ($pending_coments_count <= 5000) {

				$spam_comments       = get_comments(['status' => 'spam', 'number' => 5000]);
				$spam_comments_count = count($spam_comments);
				$data                = array_merge($pending_coments, $spam_comments);

				if ($spam_comments_count == 0) {
					$trash_coments = get_comments(['status' => 'trash', 'number' => 5000]);
					$data          = array_merge($pending_coments, $trash_coments);
				}

			} else {
				$data = $pending_coments;
			}

			$comments = [];
			foreach ($data as $comment) {

				$comments[] = [
					'comment_id' => $comment->comment_ID
				];

			}

			if (!empty($comments)) {

				$comment_id_ar = $comments;

				if ($comment_id_ar) {
					$c = '';
					foreach ($comment_id_ar as $id) {
						$c .= $id['comment_id'] . ',';
					}
					$c     = rtrim($c, ',');
					$query = 'DELETE FROM ' . $sTable . ' WHERE comment_ID IN (' . $c . ')';
					$data  = $wpdb->query($query);
				}

				if ($data) {
					XAG_Init::json('success', 'Successfully deleted comment.');
				} else {
					XAG_Init::json('error', 'Could not delete comment at the moment.');
				}

			} else {
				XAG_Init::json('error', 'Oops! Could not delete comment at the moment.');
			}

		}

		/**
		 * Delete all spam comments
		 */
		public static function deleteAllSpamComments()
		{

			global $wpdb;

			/* DB table to use */
			$sTable = $wpdb->prefix . 'comments';

			$pending_coments       = get_comments(['status' => 'spam', 'number' => 10000]);
			$pending_coments_count = count($pending_coments);

			$data = $pending_coments;

			$comments = [];
			foreach ($data as $comment) {

				$comments[] = [
					'comment_id' => $comment->comment_ID
				];

			}

			if (!empty($comments)) {

				$comment_id_ar = $comments;

				if ($comment_id_ar) {
					$c = '';
					foreach ($comment_id_ar as $id) {
						$c .= $id['comment_id'] . ',';
					}
					$c     = rtrim($c, ',');
					$query = 'DELETE FROM ' . $sTable . ' WHERE comment_ID IN (' . $c . ')';
					$data  = $wpdb->query($query);
				}

				if ($data) {
					XAG_Init::json('success', 'Successfully deleted comment.');
				} else {
					XAG_Init::json('error', 'Could not delete comment at the moment.');
				}

			} else {
				XAG_Init::json('error', 'Oops! Could not delete comment at the moment.');
			}

		}

		/**
		 * Delete all trash comments
		 */
		public static function deleteAllTrashComments()
		{

			global $wpdb;

			/* DB table to use */
			$sTable = $wpdb->prefix . 'comments';

			$pending_coments       = get_comments(['status' => 'trash', 'number' => 10000]);
			$pending_coments_count = count($pending_coments);

			$data = $pending_coments;

			$comments = [];
			foreach ($data as $comment) {

				$comments[] = [
					'comment_id' => $comment->comment_ID
				];

			}

			if (!empty($comments)) {

				$comment_id_ar = $comments;

				if ($comment_id_ar) {
					$c = '';
					foreach ($comment_id_ar as $id) {
						$c .= $id['comment_id'] . ',';
					}
					$c     = rtrim($c, ',');
					$query = 'DELETE FROM ' . $sTable . ' WHERE comment_ID IN (' . $c . ')';
					$data  = $wpdb->query($query);
				}

				if ($data) {
					XAG_Init::json('success', 'Successfully deleted comment.');
				} else {
					XAG_Init::json('error', 'Could not delete comment at the moment.');
				}

			} else {
				XAG_Init::json('error', 'Oops! Could not delete comment at the moment.');
			}

		}

		/**
		 *  Sync reviews
		 * @param boolean $array Set crone to true to return array
		 * @return void|array
		 */
		public static function getReviews($array = FALSE)
		{
			$results = MXAG_Reviews::getAllData();

			for ($i = 0; $i < sizeof($results); $i++) {
				if ($results[$i]['page_id'] !== 0) {
					$results[$i]['page_name'] = get_the_title($results[$i]['page_id']);
				} else {
					$results[$i]['page_name'] = 'Global Review';
				}
			}

			/*Returning results*/
			if ($array === TRUE) {
				return $results;
			} else {
				XAG_Init::json('success', 'Successfully retrieved reviews.', $results);
			}

		}

		public static function getComment()
		{

			$comment_id = $_POST['comment_id'];

			if (!isset($comment_id) || empty($comment_id)) {
				XAG_Init::json('error', 'Invalid request!');
			}

			$comment = get_comment($comment_id, ARRAY_A);

			if (is_null($comment)) {
				XAG_Init::json('error', 'Comment not found!');
			} else {
				XAG_Init::json('success', 'Successfully replied on comment.', $comment);
			}
		}

		public static function editComment()
		{

			$args   = $_POST['args'];
			$result = wp_update_comment($args);

			if ($result) {
				XAG_Init::json('success', 'Successfully edited comment.');
			} else {
				XAG_Init::json('error', 'Could not edit comment at the moment.');
			}

		}

		public static function replyOnComment()
		{

			$comment_id = $_POST['comment_id'];
			$content    = $_POST['content'];

			if (empty($comment_id) || empty($content)) {
				XAG_Init::json('error', 'Invalid request!');
			}

			$comment = get_comment($comment_id, ARRAY_A);

			if (is_null($comment)) {
				XAG_Init::json('error', 'Comment not found!');
			} else {

				$admins    = get_users([
					'role'   => 'administrator',
					'fields' => ['ID']
				]);
				$user_info = get_userdata($admins[0]->ID);

				$data = [
					'comment_post_ID'      => $comment['comment_post_ID'],
					'comment_author'       => $user_info->user_login,
					'comment_author_email' => $user_info->user_email,
					'comment_content'      => $content,
					'comment_type'         => '1',
					'comment_parent'       => $comment['comment_ID'],
					'user_id'              => $user_info->ID,
					'comment_date'         => current_time('mysql')
				];

				$result = wp_insert_comment($data);

				if ($result) {
					XAG_Init::json('success', 'Successfully replied on comment.');
				} else {
					XAG_Init::json('error', 'Could reply on comment at the moment.');
				}

			}

		}

		public static function updateComments()
		{
			global $wpdb;
			/* DB table to use */
			$sTable = $wpdb->prefix . 'comments';

			if (!isset($_POST['method']) || !isset($_POST['comment_id'])) {
				XAG_Init::json('error', 'Invalid request!');
			}

			$multi = is_array($_POST['comment_id']);
			// Determine the method and Comment ID
			$METHOD     = $_POST['method'];
			$comment_id = $_POST['comment_id'];

			switch ($METHOD) {
				case "PUT":

					if ($multi) {
						$c = '';
						foreach ($comment_id as $id) {
							$c .= $id . ',';
						}
						$c     = rtrim($c, ',');
						$query = 'UPDATE ' . $sTable . ' SET comment_approved = "1" WHERE comment_ID IN (' . $c . ')';
						$data  = $wpdb->query($query);

					} else {
						$data = wp_set_comment_status($comment_id, 'approve');
					}

					if ($data) {
						XAG_Init::json('success', 'Successfully Approved comment.');
					} else {
						XAG_Init::json('error', 'Could not Approve comment at the moment.');
					}

					break;
				case "POST":

					if ($multi) {
						$c = '';
						foreach ($comment_id as $id) {
							$c .= $id . ',';
						}
						$c     = rtrim($c, ',');
						$query = 'UPDATE ' . $sTable . ' SET comment_approved = "0" WHERE comment_ID IN (' . $c . ')';
						$data  = $wpdb->query($query);

					} else {
						$data = wp_set_comment_status($comment_id, 'hold');
					}

					if ($data) {
						XAG_Init::json('success', 'Successfully unapproved comment.');
					} else {
						XAG_Init::json('error', 'Could not Unapprove comment at the moment.');
					}

					break;
				case "GET":

					if ($multi) {
						$c = '';
						foreach ($comment_id as $id) {
							$c .= $id . ',';
						}
						$c     = rtrim($c, ',');
						$query = 'UPDATE ' . $sTable . ' SET comment_approved = "spam" WHERE comment_ID IN (' . $c . ')';
						$data  = $wpdb->query($query);

					} else {
						$data = wp_set_comment_status($comment_id, 'spam');
					}

					if ($data) {
						XAG_Init::json('success', 'Comment successfully moved to Spam');
					} else {
						XAG_Init::json('error', 'Could not move comment to spam at the moment.');
					}

					break;
				case "DELETE":
					if ($multi) {
						$c = '';
						foreach ($comment_id as $id) {
							$c .= $id . ',';
						}
						$c     = rtrim($c, ',');
						$query = 'UPDATE ' . $sTable . ' SET comment_approved = "trash" WHERE comment_ID IN (' . $c . ')';
						$data  = $wpdb->query($query);

					} else {
						$data = wp_set_comment_status($comment_id, 'trash');
					}

					if ($data) {
						XAG_Init::json('success', 'Successfully deleted comment.');
					} else {
						XAG_Init::json('error', 'Could not delete comment at the moment.');
					}

					break;
			}

		}

		/**
		 *   Get Posts
		 * @param boolean $array Set cron to true to return array
		 * @return void|array
		 */
		public static function getPosts($array = FALSE)
		{

			global $wpdb;

			$aColumns = [
				'ID',
				'post_author',
				'post_date',
				'post_title',
				'post_status',
				'comment_status',
				'post_name',
				'post_parent',
				'guid',
				'post_type',
				'comment_count'
			];

			/* Indexed column (used for fast and accurate table cardinality) */
			$sIndexColumn = "ID";

			/* DB table to use */
			$sTable = $wpdb->prefix . 'posts';

			// Determine Post Types
			$allowedPostTypes = MXAG_Seo::getAllPostTypes();

			$sWhere = " WHERE post_type IN ('" . join("','", $allowedPostTypes) . "') ";

			$sWhere .= " AND post_status IN ('publish', 'future', 'draft', 'pending') ";

			/*
			 * SQL queries
			 * Get data to display
			 */
			$sQuery = "
				SELECT SQL_CALC_FOUND_ROWS " . str_replace(" , ", " ", implode(", ", $aColumns)) . "
				FROM $sTable
				$sWhere";

			$rResult = $wpdb->get_results($sQuery, ARRAY_A);

			$posts_data = [];

			for ($i = 0; $i < count($rResult); $i++) {
				$posts_data[$i]['id']            = $rResult[$i]['ID'];
				$posts_data[$i]['post_type']     = $rResult[$i]['post_type'];
				$posts_data[$i]['post_title']    = $rResult[$i]['post_title'];
				$posts_data[$i]['post_status']   = $rResult[$i]['post_status'];
				$posts_data[$i]['guid']          = $rResult[$i]['guid'];
				$posts_data[$i]['post_author']   = get_the_author_meta('display_name', $rResult[$i]['post_author']);
				$posts_data[$i]['comment_count'] = $rResult[$i]['comment_count'];
			}

			/*Returning results*/
			if ($array === TRUE) {
				return $posts_data;
			} else {
				XAG_Init::json('success', 'Successfully retrieved all post revisions.', $posts_data);
			}

		}

		/**
		 *   Get Post Revisions
		 * @param boolean $array Set cron to true to return array
		 * @return void|array
		 */
		public static function getPostRevisions($array = FALSE)
		{
			$posts = get_posts();

			$post_ids = [];
			foreach ($posts as $post) {
				$post_ids[] = $post->ID;
			}

			$revisions = [];
			foreach ($post_ids as $id) {
				$revision_count = sizeof(wp_get_post_revisions($id));

				if ($revision_count > 0) {
					$revisions[$id]['count'] = $revision_count;
				}
			}

			/*Returning results*/
			if ($array === TRUE) {
				return $revisions;
			} else {
				XAG_Init::json('success', 'Successfully retrieved all post revisions.', $revisions);
			}

		}

		public static function deleteRevisions()
		{
			if (!isset($_POST['method']) || !isset($_POST['post_id'])) {
				XAG_Init::json('error', 'Invalid request!');
			}

			// Determine the method and Comment ID
			$METHOD  = $_POST['method'];
			$post_id = trim($_POST['post_id']);

			switch ($METHOD) {
				case "DELETE":

					$revisions = wp_get_post_revisions($post_id);

					$results = [];
					foreach ($revisions as $revision) {
						$results[$revision->ID] = wp_delete_post_revision($revision->ID);
					}

					foreach ($results as $result) {
						if (is_wp_error($result)) {
							XAG_Init::json('error', 'Could not delete all revisions.');
							die();
						}
					}

					XAG_Init::json('success', 'Successfully deleted revisions.');

					break;
				default:
					XAG_Init::json('error', 'Invalid request!');
					break;
			}

		}

		/**
		 * Update Features
		 */
		public static function updateFeatures()
		{

			// Update features if detected
			if (isset($_POST['features']) && !empty($_POST['features'])) {
				update_option('ps_features', $_POST['features']);
				XAG_Init::json('success', 'Successfully updated features!');
			} else {

				XAG_Init::json('error', 'Features not detected!');

			}

		}

		/**
		 *   Daily Synchronization
		 */
		public static function dailySync()
		{

			try {

				$received_features = FALSE;

				// Update features if detected
				if (isset($_POST['features']) && !empty($_POST['features'])) {
					$received_features = $_POST['features'];
					update_option('ps_features', $_POST['features']);
				}

				// Update features if detected
				if (isset($_POST['membership'])) {
					update_option('ps_user_membership', $_POST['membership']);
				}

				// deactivate on panel if license not set
	            if (!XAG_Licencator::isLicenseSet()) {
	                XAG_Init::deactivate();
	            }

				XAG_Init::json('success', 'Successfully retrieved blog description.', [

					/*Posts*/
					'posts'            => self::getPosts(TRUE),

					/*Plugins and Themes*/
					'pluginsAndThemes' => self::storePluginsThemes(TRUE),

					/*Available Updates*/
					'availableUpdates' => self::getUpdates(TRUE),

					/*Settings*/
					'settings'         => self::getSettings(TRUE),

					/*Comments*/
					'comments'         => self::getComments(TRUE),					

					/*Settings*/
					'revisions'        => self::getPostRevisions(TRUE),

					/*Reviews*/
					'reviews'          => self::getReviews(TRUE),

					/*WordPress Version*/
					'coreVersion'      => get_bloginfo('version'),

					/*Send back received features*/
					'new_features'     => $received_features,
					'old_features'     => get_option('ps_features'),

					/*Update Admin Post*/
					'admin_post'       => (get_option('prs_ts_alternative_api') == TRUE) ? get_site_url() . '/xagio-api.php' : admin_url() . 'admin-post.php'
				]);

			} catch (Exception $error) {
				XAG_Init::json('error', 'Plugin error occured. Please pass this down to support: ' . $error->getMessage());
			}

		}

		/**
		 *   Rank Tracker Daily Synchronization
		 */
		public static function dailySyncPRT()
		{

			try {

				// Check if prtData is set
				if (!isset($_POST['prtData'])) {
					XAG_Init::json('error', 'Invalid request!');
				}

				// Clear all fields
				global $wpdb;
				$query = 'UPDATE prs_keywords SET rank = "0"';
				$wpdb->query($query);


				// Update existing fields
				foreach ($_POST['prtData'] as $termName => $termData) {
					MXAG_Model::updateData([
						'rank' => json_encode($termData, TRUE)
					], [
						'keyword' => $termName
					], "prs_keywords");
				}

			} catch (Exception $error) {
				XAG_Init::json('error', 'Plugin error occured. Please pass this down to support: ' . $error->getMessage());
			}

		}

		/**
		 *   Rank Tracker update remove
		 */
		public static function removeTermPRT()
		{
			// Check if keywords are set
			if (!isset($_POST['term_ids'])) {
				XAG_Init::json('error', 'Invalid request!');
			}

			$term_ids = explode(',', $_POST['term_ids']);
			global $wpdb;

			foreach ($term_ids as $term_id) {
				$results = $wpdb->get_results("SELECT keyword,rank FROM prs_keywords WHERE rank != '0' AND rank != '501' AND rank LIKE '%$term_id%' LIMIT 1");
				$results = $results[0];
				$data    = json_decode($results->rank, TRUE);
				$newData = [];

				foreach ($data as $key => $val) {
					if ($val['term_id'] == $term_id) {
						unset($data[$key]);
					} else {
						$newData[] = $val;
					}
				}

				if (sizeof($newData) !== 0) {
					$newData = json_encode($newData, TRUE);
				} else {
					$newData = '0';
				}

				MXAG_Model::updateData([
					'rank' => $newData
				], [
					'keyword' => $results->keyword
				], "prs_keywords");
			}
		}


		/**
		 *   Store Plugins and Themes on register
		 * @param boolean $array Set crone to true to return array
		 * @return void|array
		 */
		public static function storePluginsThemes($array = FALSE)
		{

			$pluginsThemes = self::getPluginsThemes(TRUE);

			// Check if plugins are activated
			$plugins = $pluginsThemes['plugins'];

			foreach ($plugins as $name => $plugin) {
				$pluginsThemes['plugins'][$name]['active'] = is_plugin_active($name);
			}

			// Check which theme is activated
			$themes = $pluginsThemes['themes'];

			$current_theme = get_option('current_theme');

			foreach ($themes as $name => $theme) {
				if ($theme['Name'] == $current_theme) {
					$pluginsThemes['themes'][$name]['active'] = TRUE;
				} else {
					$pluginsThemes['themes'][$name]['active'] = FALSE;
				}
			}

			/*Returning results*/
			if ($array === TRUE) {
				return $pluginsThemes;
			} else {
				XAG_Init::json('success', 'Successfully retrieved blog description.', $pluginsThemes);
			}

		}

		/**
		 *  Get site Environment
		 */
		public static function getEnvironment()
		{
			$wpDebugMode = (defined('WP_DEBUG') && TRUE === WP_DEBUG) ? TRUE : FALSE;
			$data        = [
				'wpEnvironment'     => [
					'homeURL'       => get_home_url(),
					'siteURL'       => get_site_url(),
					'wpVersion'     => get_bloginfo('version'),
					'wpMultisite'   => is_multisite(),
					'wpMemoryLimit' => WP_MEMORY_LIMIT,
					'wpDebugMode'   => $wpDebugMode,
					'wpLanguage'    => get_bloginfo('language')
				],
				'serverEnvironment' => [
					'serverInfo'       => $_SERVER['SERVER_SOFTWARE'],
					'phpVersion'       => phpversion(),
					'phpPostMaxSize'   => ini_get('post_max_size'),
					'phpMaxUploadSize' => ini_get('upload_max_filesize'),
					'phpTimeLimit'     => ini_get('max_execution_time'),
					'phpMaxInputVars'  => ini_get('max_input_vars'),
					'cURL'             => function_exists('curl_init'),
					'ZipArchive'       => class_exists('ZipArchive'),
					'DOMDocument'      => class_exists('DOMDocument'),
					'wpRemoteGet'      => function_exists('wp_remote_get'),
					'wpRemotePost'     => function_exists('wp_remote_post')
				]
			];

			XAG_Init::json('success', 'Successfully retrieved Environment info.', $data);
		}

		/**
		 *  Save Global Scripts
		 */
		public static function renderGlobalScripts()
		{

			$page_id = $_POST['page_id'];
			$value   = $_POST['value'];
			update_post_meta($page_id, 'ps_seo_disable_global_scripts', @$value);
			XAG_Init::json('success', 'Successfully updated render global script option.');

		}

		/**
		 *  Get Scripts per page
		 */
		public static function getPerPageScript()
		{

			$page_id = $_POST['page_id'];

			$scripts = get_post_meta($page_id, 'ps_seo_scripts', TRUE);
			$scripts = base64_encode($scripts);

			XAG_Init::json('success', 'Successfully retrieved scripts.', $scripts);
		}

		/**
		 *  Save Global Scripts
		 */
		public static function savePerPageScript()
		{

			$page_id = $_POST['page_id'];
			$encoded = $_POST['script'];
			$script  = base64_decode($encoded);
			update_post_meta($page_id, 'ps_seo_scripts', trim($script));

			XAG_Init::json('success', 'Successfully updated global script.');

		}

		/**
         *  Get global header Scripts
         */
        public static function getGlobalScripts() {

            $scripts = get_option('ps_seo_global_scripts');

            $scripts = base64_encode($scripts);

            XAG_Init::json( 'success', 'Successfully retrieved header scripts.', $scripts);
        }

        /**
         *  Get global footer Scripts
         */
        public static function getGlobalFooterScripts() {

            $scripts = get_option('ps_seo_global_footer_scripts');

            $scripts = base64_encode($scripts);

            XAG_Init::json( 'success', 'Successfully retrieved footer scripts.', $scripts);
        }

        /**
         *  Save Global header Scripts
         */
        public static function saveGlobalScript() {

            $encoded_header = $_POST['script']['global_header_scripts'];
            $encoded_footer = @$_POST['script']['global_footer_scripts'];
            $header_script = base64_decode($encoded_header);
            $footer_script = base64_decode($encoded_footer);

            update_option('ps_seo_global_scripts', trim($header_script));
            update_option('ps_seo_global_footer_scripts', trim($footer_script));

            XAG_Init::json( 'success', 'Successfully updated global scripts.');

        }

        /**
         *  Save Global footer Scripts
         */
        public static function saveGlobalFooterScript() {

            $encoded = $_POST['footerScript'];
            $script = base64_decode($encoded);

            update_option('ps_seo_global_footer_scripts', trim($script));

            XAG_Init::json( 'success', 'Successfully updated global footer script.');

        }

		/**
		 *  Update WordPress Core
		 */
		public static function updateWpCore()
		{

			try {

				// Remove the lock
				delete_option('core_updater.lock');

				// Including required class for updates
				require_once(ABSPATH . 'wp-admin/includes/class-wp-upgrader.php');

				$reinstall = isset($_POST['reinstall']);
				$version   = isset($_POST['version']) ? $_POST['version'] : FALSE;

				// Check if already updated
				if ($version == get_bloginfo('version')) {
					XAG_Init::json('success', 'Successfully updated WordPress core.');
					exit;
				}

				$update = find_core_update($version, 'en_US');

				if (!$update) {
					XAG_Init::json('error', 'Cannot find any new updates for WordPress core.');
					exit;
				}

				$allow_relaxed_file_ownership = !$reinstall && isset($update->new_files) && !$update->new_files;

				if ($reinstall) {
					$update->response = 'reinstall';
				}

				ob_start();

				$upgrader = new Core_Upgrader();
				$status   = $upgrader->upgrade($update, [
					'allow_relaxed_file_ownership' => $allow_relaxed_file_ownership
				]);

				ob_end_clean();

				if (is_wp_error($status)) {
					XAG_Init::json('error', 'Failed to update WordPress core.', $status);
				} else {
					XAG_Init::json('success', 'Successfully updated WordPress core.');
				}

			} catch (Exception $error) {
				XAG_Init::json('error', 'Plugin error occured. Please pass this down to support: ' . $error->getMessage());
			}

		}

		/**
		 * Update one or more plugins
		 * receive plugin root_name
		 */
		public static function updateWpPlugins()
		{

			try {

				// Check if is set plugin name
				if (!isset($_POST['pluginNames'])) {
					XAG_Init::json('error', 'Invalid request!');
				}

				$plugins = $_POST['pluginNames'];

				if (is_array($plugins)) {
					$pluginNames = $plugins;
				} else {
					$pluginNames = [$plugins];
				}

				// Including required class for updates
				require_once(ABSPATH . 'wp-admin/includes/class-wp-upgrader.php');

				$status      = [];
				$all_plugins = get_plugins();

				foreach ($pluginNames as $pluginName) {

					$updates        = self::getUpdates(TRUE);
					$plugins_update = $updates['plugins'];

					if (!isset($all_plugins[$pluginName])) {
						$status[$pluginName] = 'Plugin doesn\'t exist. Please synchronize.';
					} else {

						if (!isset($plugins_update[$pluginName])) {
							$status[$pluginName] = TRUE;
						} else {
							$updater    = new Plugin_Upgrader();
							$temp_check = $updater->bulk_upgrade([$pluginName]);

							if (!$temp_check) {
								$status[$pluginName] = FALSE;
							} else {
								$status[$pluginName] = TRUE;
							}
						}
					}
				}

				XAG_Init::json('success', 'Operation successfully finished.', ['success' => $status]);

			} catch (Exception $error) {
				XAG_Init::json('error', 'Plugin error occured. Please pass this down to support: ' . $error->getMessage());
			}

		}

		/**
		 * Update one or more themes
		 * theme slug required
		 */
		public static function updateWpThemes()
		{

			try {

				// Check if is set plugin name
				if (!isset($_POST['themeNames'])) {
					XAG_Init::json('error', 'Invalid request!');
				}

				$themes = $_POST['themeNames'];

				if (is_array($themes)) {
					$themeNames = $themes;
				} else {
					$themeNames = [$themes];
				}

				// Including required class for updates
				require_once(ABSPATH . 'wp-admin/includes/class-wp-upgrader.php');

				$status = [];

				$updates        = self::getUpdates(TRUE);
				$themes_updates = $updates['themes'];

				foreach ($themeNames as $themeName) {
					if (!isset($themes_updates[$themeName])) {
						$status[$themeName] = TRUE;
					} else {
						$updater            = new Theme_Upgrader();
						$status[$themeName] = $updater->upgrade($themeName);
					}
				}

				XAG_Init::json('success', 'Operation successfully finished.', $status);

			} catch (Exception $error) {
				XAG_Init::json('error', 'Plugin error occured. Please pass this down to support: ' . $error->getMessage());
			}

		}

		/**
		 * Removing theme from WordPress
		 * theme slug required
		 */
		public static function removeWpTheme()
		{

			try {

				if (!isset($_POST['slug'])) {
					XAG_Init::json('error', 'Invalid request!');
				}

				require_once(ABSPATH . 'wp-admin/includes/theme.php');
				require_once(ABSPATH . 'wp-admin/includes/file.php');

				$remove = delete_theme($_POST['slug']);

				if ($remove == TRUE || $remove == 1) {
					XAG_Init::json('success', 'Theme successfully removed.');
				} else {
					XAG_Init::json('error', 'There was some problem while removing theme', $remove);
				}

			} catch (Exception $error) {
				XAG_Init::json('error', 'Plugin error occured. Please pass this down to support: ' . $error->getMessage());
			}
		}

		/**
		 * Create new Page/Post from Articles
		 * */
		public static function cuArticlesPagePost()
		{

			if (!isset($_POST['pagePostData'])) {
				XAG_Init::json('error', 'Invalid request!');
			}


			$page           = $_POST['pagePostData'];
			$replaceContent = isset($_POST['replaceContent']) ? $_POST['replaceContent'] : 0;

			if (isset($page['ID'])) {

				if ($replaceContent == 0) {
					$post_content         = get_post($page['ID']);
					$content              = $post_content->post_content;
					$page['post_content'] = $content . '<br>' . $page['post_content'];
				}

				// update
				$page_id = wp_update_post($page);

				if ($page_id == 0) {
					XAG_Init::json('error', 'There was a problem while updating!');
				} else {
					XAG_Init::json('success', 'Successfully updated!');
				}

			} else {
				// create
				$page['post_parent'] = 0;
				$page['post_author'] = 1;

				$page_id = wp_insert_post($page);

				if ($page_id == 0) {
					XAG_Init::json('error', 'There was a problem while creating!');
				} else {
					XAG_Init::json('success', 'Successfully created!');
				}
			}

		}

		/**
		 *  Approve / Edit / Remove Comment
		 */
		public static function deleteComment()
		{
			$comment_id = intval($_POST['id']);

			$data = wp_delete_comment($comment_id);

			if ($data) {
				XAG_Init::json('success', 'Successfully deleted Comment.');
			} else {
				XAG_Init::json('error', 'Comment doesn\'t exist!');
			}
		}

		public static function approveComment()
		{
			$comment_id = intval($_POST['id']);

			$data = wp_set_comment_status($comment_id, 'approve');

			if ($data) {
				XAG_Init::json('success', 'Successfully approved Comment.');
			} else {
				XAG_Init::json('error', 'Comment doesn\'t exist!');
			}
		}

		/**
		 *  Edit / Add Review
		 */
		public static function editAddReview()
		{
			$ID = intval($_POST['id']);

			$data = [
				'name'      => sanitize_text_field($_POST['name']),
				'review'    => sanitize_text_field($_POST['review']),
				'rating'    => intval($_POST['rating']),
				'email'     => sanitize_email($_POST['email']),
				'website'   => esc_url_raw($_POST['website']),
				'telephone' => sanitize_text_field($_POST['telephone']),
				'location'  => sanitize_text_field($_POST['location']),
				'age'       => intval($_POST['name'])
			];

			if ($ID == 0) {
				MXAG_Reviews::insertData($data);
			} else {
				MXAG_Reviews::updateData($data, [
					'id' => $ID
				]);
			}
			XAG_Init::json('success', 'Successfully finished operation.');
		}

		/**
		 * Syn reviews
		 * */
		public static function updateReviews()
		{

			if (!isset($_POST['method']) || !isset($_POST['review_ids'])) {
				XAG_Init::json('error', 'Invalid request!');
			}

			// Determine the method and Review ID
			$METHOD     = $_POST['method'];
			$review_ids = trim($_POST['review_ids']);

			$review_ids = explode(',', $review_ids);

			switch ($METHOD) {
				case "approve":
					MXAG_Reviews::updateData([
						'approved' => 1
					], [
						'id' => $review_ids
					]);
					break;
				case "unapprove":
					MXAG_Reviews::updateData([
						'approved' => 0
					], [
						'id' => $review_ids
					]);
					break;
				case "remove":
					MXAG_Reviews::removeData([
						'id' => $review_ids
					]);
					break;
				default:
					XAG_Init::json('error', 'Invalid request!');
					break;
			}

			XAG_Init::json('success', 'Operation completed.');

		}

		/**
		 *  Syn posts
		 */
		public static function updatePosts()
		{

			if (!isset($_POST['method']) || !isset($_POST['post_ids'])) {
				XAG_Init::json('error', 'Invalid request!');
			}

			// Determine the method and Review ID
			$METHOD   = $_POST['method'];
			$post_ids = trim($_POST['post_ids']);

			$post_ids = explode(',', $post_ids);

			switch ($METHOD) {
				case "seo-enable":
					foreach ($post_ids as $id) {
						update_post_meta($id, 'ps_seo_enabled', TRUE);
					}
					break;
				case "seo-disable":
					foreach ($post_ids as $id) {
						update_post_meta($id, 'ps_seo_enabled', FALSE);
					}
					break;
				case "remove":
					foreach ($post_ids as $id) {
						wp_trash_post($id);
					}
					break;
				default:
					XAG_Init::json('error', 'Invalid request!');
					break;
			}

			XAG_Init::json('success', 'Operation completed.');

		}


		/**
		 *  Sync settings
		 * @param boolean $array Set crone to true to return array
		 * @return void|array
		 */
		public static function getSettings($array = FALSE)
		{
			$settings = ["blogname", "blogdescription", "users_can_register", "default_role", "default_category", "default_post_format", "show_on_front", "page_on_front", "page_for_posts", "posts_per_page", "rss_use_excerpt", "blog_public", "default_pingback_flag", "default_ping_status", "default_comment_status", "require_name_email", "comment_registration", "close_comments_for_old_posts", "close_comments_days_old", "thread_comments", "thread_comments_depth", "page_comments", "comments_per_page", "default_comments_page", "comment_order", "comments_notify", "moderation_notify", "comment_moderation", "comment_whitelist", "permalink_structure"];

			$setting_for_panel = [];

			foreach ($settings as $setting) {
				$setting_for_panel[$setting] = get_option($setting);
			}

			$categories = get_categories([
					"hide_empty" => 0,
					"type"       => "post",
					"orderby"    => "name",
					"order"      => "ASC"
				]
			);
			$cat_data   = [];
			foreach ($categories as $category) {
				$cat_data[] = [
					"term_id"         => $category->term_id,
					"name"            => $category->name,
					"slug"            => $category->slug,
					"category_parent" => $category->category_parent
				];
			}

			$setting_for_panel["categories"] = json_encode($cat_data, TRUE);

			/*Returning results*/
			if ($array === TRUE) {
				return $setting_for_panel;
			} else {
				XAG_Init::json('success', 'Successfully retrieved blog description.', $setting_for_panel);
			}

		}

		public static function post_settings()
		{
			$DATA = (isset($_POST['data'])) ? (!empty($_POST['data'])) ? $_POST['data'] : 0 : 0;
			if ($DATA == 0) {
				XAG_Init::json('error', 'DATA is not properly being sent.');
			} else {
				foreach ($DATA as $option => $val) {
					update_option($option, $val);
				}
				XAG_Init::json('success', 'DATA successfully updated.');
			}
		}

		public static function pushSchema()
		{
			$schemaIDs = (isset($_POST['schema_ids'])) ? (!empty($_POST['schema_ids'])) ? explode(',', $_POST['schema_ids']) : NULL : NULL;
			$ID        = (isset($_POST['id'])) ? (!empty($_POST['id'])) ? trim($_POST['id']) : 0 : 0;
			$TYPE      = (isset($_POST['type'])) ? (!empty($_POST['type'])) ? trim($_POST['type']) : 'post' : 'post';

			if (!empty($schemaIDs)) {

				$output          = NULL;
				$renderedSchemas = MXAG_Schema::getRemoteRenderedSchemas($schemaIDs, NULL, 'post', $output);

				if ($renderedSchemas != FALSE) {

					if ($TYPE == 'post') {

						if ($ID !== 0) {
							update_post_meta($ID, 'ps_schema_meta', $renderedSchemas['meta']);
							update_post_meta($ID, 'ps_schema_data', $renderedSchemas['data']);
						} else {
							update_option('ps_schema_meta', $renderedSchemas['meta']);
							update_option('ps_schema_data', $renderedSchemas['data']);
						}

					} else if ($TYPE == 'term') {

						if ($ID !== 0) {

							$tag  = get_term($ID);
							$id   = $tag->term_id;
							$tax  = $tag->taxonomy;
							$meta = get_option($tax . '_' . $id);

							if (empty($meta)) {
								$meta = [];
							}

							$meta['ps_schema_meta'] = $renderedSchemas['meta'];
							$meta['ps_schema_data'] = $renderedSchemas['data'];

							update_option($tax . '_' . $id, $meta);

						} else {
							// TODO -- Add global term schema
						}

					}

					// Clear the caches
					MXAG_Api::clearCaches();

					XAG_Init::json('success', 'Operation successfully finished.');
				} else {
					XAG_Init::json('error', 'A problem occurred.', $output);
				}

			} else {
				if ($ID !== 0) {
					update_post_meta($ID, 'ps_schema_meta', FALSE);
					update_post_meta($ID, 'ps_schema_data', FALSE);
				} else {
					update_option('ps_schema_meta', FALSE);
					update_option('ps_schema_data', FALSE);
				}
				XAG_Init::json('success', 'Operation successfully finished.');
			}
		}

		public static function getSchema($return = FALSE)
		{

			$ID   = (isset($_POST['id'])) ? (!empty($_POST['id'])) ? trim($_POST['id']) : 0 : 0;
			$TYPE = (isset($_POST['type'])) ? (!empty($_POST['type'])) ? trim($_POST['type']) : 'post' : 'post';

			if ($TYPE == 'post') {

				$SCHEMA = ($ID !== 0) ? get_post_meta($ID, 'ps_schema_meta', TRUE) : get_option('ps_schema_meta');

			} else if ($TYPE == 'term') {

				$tag  = get_term($ID);
				$id   = $tag->term_id;
				$tax  = $tag->taxonomy;
				$meta = get_option($tax . '_' . $id);

				$SCHEMA = $meta['ps_schema_meta'];
			}

			if ($return === TRUE) {
				return $SCHEMA;
			} else {
				XAG_Init::json('success', 'Retrieved schema(s).', $SCHEMA);
			}
		}

		public static function removeSchema()
		{
			$schema_id = (isset($_POST['schema_id'])) ? (!empty($_POST['schema_id'])) ? trim($_POST['schema_id']) : NULL : NULL;
			$page_ids  = (isset($_POST['page_ids'])) ? (!empty($_POST['page_ids'])) ? $_POST['page_ids'] : [] : [];


			if ($schema_id === NULL || empty($page_ids)) {
				XAG_Init::json('error', 'Not valid ID');
			} else {

				foreach ($page_ids as $page_id) {

					if ($page_id === 0) {
						$schema_meta = get_option('ps_schema_meta');

						if (!empty($schema_meta)) {
							$index = NULL;

							for ($i = 0; $i < sizeof($schema_meta); $i++) {
								$meta = $schema_meta[$i];

								if ($meta['id'] === $schema_id) {
									$index = $i;
									unset($schema_meta[$i]);

									update_option('ps_schema_meta', $schema_meta);
									break;
								}
							}

							if ($index !== NULL) {
								$schema_data = get_option('ps_schema_data');

								if (isset($schema_data[$index])) {
									unset($schema_data[$index]);

									update_option('ps_schema_data', $schema_data);
								}
							}
						}

						continue;
					}

					$schema_meta = get_post_meta($page_id, 'ps_schema_meta', TRUE);

					if (!empty($schema_meta)) {
						$index = NULL;

						for ($i = 0; $i < sizeof($schema_meta); $i++) {
							$meta = $schema_meta[$i];

							if ($meta['id'] === $schema_id) {
								$index = $i;
								unset($schema_meta[$i]);

								update_post_meta($page_id, 'ps_schema_meta', $schema_meta);
								break;
							}
						}

						if ($index !== NULL) {
							$schema_data = get_post_meta($page_id, 'ps_schema_data', TRUE);

							if (isset($schema_data[$index])) {
								unset($schema_data[$index]);

								update_post_meta($page_id, 'ps_schema_data', $schema_data);
							}
						}
					}

				}
				XAG_Init::json('success', 'Removed successfully');
			}
		}

		public static function getPost()
		{
			$ID = (isset($_POST['id'])) ? (!empty($_POST['id'])) ? trim($_POST['id']) : 0 : 0;
			if ($ID == 0) {
				XAG_Init::json('error', 'ID is not properly being sent.');
			} else {
				$page = get_post($ID);
				XAG_Init::json('success', 'Successfully retrieved post content.', $page);
			}
		}

		public static function updatePost()
		{
			$DATA = (isset($_POST['data'])) ? (!empty($_POST['data'])) ? $_POST['data'] : 0 : 0;
			if ($DATA == 0) {
				XAG_Init::json('error', 'DATA is not properly being sent.');
			} else {
				wp_update_post($DATA);
				XAG_Init::json('success', 'Successfully updated post content.');
			}
		}

		public static function createPost()
		{
			$DATA = (isset($_POST['data'])) ? (!empty($_POST['data'])) ? $_POST['data'] : 0 : 0;
			if ($DATA == 0) {
				XAG_Init::json('error', 'DATA is not properly being sent.');
			} else {
				wp_insert_post($DATA);
				XAG_Init::json('success', 'Successfully created new ' . $DATA['post_type'] . '.');
			}
		}

		public static function toggleSEO()
		{
			$ID    = (isset($_POST['id'])) ? (!empty($_POST['id'])) ? trim($_POST['id']) : 0 : 0;
			$VALUE = (isset($_POST['value'])) ? (!empty($_POST['value'])) ? trim($_POST['value']) : 0 : 0;

			if ($ID == 0) {
				XAG_Init::json('error', 'ID is not properly being sent.');
			} else {
				update_post_meta($ID, 'ps_seo_enabled', $VALUE);
				XAG_Init::json('success', 'Successfully toggled SEO.');
			}
		}

		public static function apiRequestUpload(
			$apiEndpoint = NULL,
			$fileToUpload = NULL
		) {

			$license_email = '';
			$license_key   = '';
			if (!$license_set = XAG_Licencator::isLicenseSet($license_email, $license_key)) {
				return FALSE;
			}

			if ($apiEndpoint == NULL || $fileToUpload == NULL) {
				return FALSE;
			}

			if (function_exists('curl_file_create')) { // php 5.5+
				$cFile = curl_file_create($fileToUpload);
			} else { //
				$cFile = '@' . realpath($fileToUpload);
			}

			$post = [
				'license_email' => $license_email,
				'license_key'   => $license_key,
				'file_contents' => $cFile
			];

			// Set the domain name
			$domain = preg_replace('/^www\./', '', $_SERVER['SERVER_NAME']);

			$ch = curl_init();
			curl_setopt($ch, CURLOPT_URL, XAG_PANEL_URL . "/api/" . $apiEndpoint);
			curl_setopt($ch, CURLOPT_POST, TRUE);
			curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
			curl_setopt($ch, CURLOPT_USERAGENT, "PSv3 - " . XAG_CURRENT_VERSION . " ($domain)");
			curl_setopt($ch, CURLOPT_FOLLOWLOCATION, TRUE);
			curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
			$result = curl_exec($ch);
			curl_close($ch);

			$data = json_decode($result, TRUE);
			if (!$data) {
				return FALSE;
			} else {
				;
				return $data;
			}
		}

		public static function apiRequest(
			$apiEndpoint = NULL,
			$method = 'GET',
			$args = [],
			&$http_code = FALSE
		) {
			$license_email = '';
			$license_key   = '';
			if (!$license_set = XAG_Licencator::isLicenseSet($license_email, $license_key)) {
				return FALSE;
			}

			if ($apiEndpoint == NULL) {
				return FALSE;
			}

			// Set the domain name
			$domain = preg_replace('/^www\./', '', $_SERVER['SERVER_NAME']);

			// Set the HTTP Query
			$http_query = [
				'license_email' => $license_email,
				'license_key'   => $license_key,
			];

			$http_query = array_merge($http_query, $args);

			$response = wp_remote_post(XAG_PANEL_URL . "/api/" . $apiEndpoint, [
					'user-agent'  => "PSv3 - " . XAG_CURRENT_VERSION . " ($domain)",
					'timeout'     => 30,
					'redirection' => 5,
					'httpversion' => '1.0',
					'blocking'    => TRUE,
					'method'      => $method,
					'body'        => $http_query,
				]
			);

			if (is_wp_error($response)) {
				return FALSE;
			} else {

				$http_code = $response['response']['code'];

				if (!isset($response['body'])) {
					return FALSE;
				} else {
					$data = json_decode($response['body'], TRUE);
					if (!$data) {
						return FALSE;
					} else {
						return $data;
					}
				}
			}
		}

		public static function syncServerAPI()
		{
			do_action('xag_LicenseCheck');
		}

		private static function clearCaches()
		{

			try {

				global $wp_fastest_cache;

				// if W3 Total Cache is being used, clear the cache
				if (function_exists('w3tc_pgcache_flush')) {
					@w3tc_pgcache_flush();
				} // if WP Super Cache is being used, clear the cache
				else if (function_exists('wp_cache_clean_cache')) {
					global $file_prefix, $supercachedir;
					if (empty($supercachedir) && function_exists('get_supercache_dir')) {
						$supercachedir = @get_supercache_dir();
					}
					@wp_cache_clean_cache($file_prefix);
				} else if (class_exists('WpeCommon')) {
					//be extra careful, just in case 3rd party changes things on us
					if (method_exists('WpeCommon', 'purge_memcached')) {
						@WpeCommon::purge_memcached();
					}
					if (method_exists('WpeCommon', 'purge_memcached')) {
						@WpeCommon::clear_maxcdn_cache();
					}
					if (method_exists('WpeCommon', 'purge_memcached')) {
						@WpeCommon::purge_varnish_cache();
					}
				} else if (method_exists('WpFastestCache', 'deleteCache') && !empty($wp_fastest_cache)) {
					@$wp_fastest_cache->deleteCache();
				}

			} catch (Exception $error) {
				XAG_Init::json('error', 'Plugin error occured. Please pass this down to support: ' . $error->getMessage());
			}

		}

		private function removeDirectory($path)
		{
			$files = glob($path . '/*');
			foreach ($files as $file) {
				is_dir($file) ? self::removeDirectory($file) : unlink($file);
			}
			rmdir($path);
			return;
		}
	}
}
