<?php

if ( ! class_exists( 'MXAG_Backups' ) ) {

	class MXAG_Backups {

		public static function initialize() {
			// Disable Model if not license is present
			if (!XAG_Licencator::isLicenseSet()) return;

			// Add cron schedules
			add_filter( 'cron_schedules', array( 'MXAG_Backups', 'customSchedules') );

			// Set Crons
			add_action('xag_doBackup', array( 'MXAG_Backups', 'doBackup'));

			// Load Backup Keys
			self::loadKeys();
		}

		public static function customSchedules($schedules) {
			if(!isset($schedules["biweekly"])){
				$schedules["biweekly"] = array(
					'interval' => 1209600,
					'display' => __('Once every two weeks'));
			}
			if(!isset($schedules["monthly"])){
				$schedules["monthly"] = array(
					'interval' => 2419200,
					'display' => __('Once every month'));
			}
			return $schedules;
		}

		// Function that handles the restoration of backups remotely
		public static function restoreBackupHandler($storage, $backup, $backup_id) {
			// Get the needed AccessTokens
			$tokens = get_option('ps_backup_settings');

			// Give PHP enough time to restore the backup
			@set_time_limit(0);

			// Check whether the folders exist (if not, make them)
			$backupFolder = XAG_PATH . '/backups';
			if ( ! file_exists( $backupFolder ) ) {
				mkdir( $backupFolder );
			}

			// Include the ZIP Class
			$zipClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'pclzip.lib.php';
			if (!class_exists('PclZip')) {
				require_once ( $zipClass );
			}

			// Create Backup Name (both folder and file name)
			$siteName = self::getName(site_url());

			// get the real site name
			$realSiteName = self::getName(site_url(), true);

			/* DROPBOX */
			if ($storage === 'dropbox') {
				// Include the DropBox Class
				$dropBoxClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'DropboxClient.php';
				if ( ! class_exists( 'DropboxClient' ) ) {
					require_once( $dropBoxClass );
				}

				// check the token
				$backup_DropboxAccessToken = isset($tokens['dropbox']['access_token']) ? $tokens['dropbox']['access_token'] : '';

				if (!empty($backup_DropboxAccessToken)) {

					if (!XAG_DROPBOX_KEY || !XAG_DROPBOX_SECRET) {
						return array(
							'status'  => 'error',
							'message' => 'Please synchronize your Backup Settings to obtain latest updated settings.'
						);
					}

					// Initialize the class
					$dropboxClient = new DropboxClient( array(
						'app_key'         => XAG_DROPBOX_KEY,
						'app_secret'      => XAG_DROPBOX_SECRET
					), $tokens['dropbox']['access_token'] );

					// Try to create a folder
					$dropboxClient->Download('/' . $siteName . '/' . $backup, $backupFolder . '/' . $backup);
				} else {
					return array(
						'status'  => 'error',
						'message' => 'Please synchronize your Backup Settings to obtain your Dropbox Access Token on '.$realSiteName.' website.'
					);
				}
			}
			/* DROPBOX */

            /* ONEDRIVE */
            if ($storage === 'onedrive') {
                // Include the OneDrive Class
                $oneDriveClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'OnedriveClient.php';
                if ( ! class_exists( 'OnedriveClient' ) ) {
                    require_once( $oneDriveClass );
                }

                // check the token
                $backup_OnedriveAccessToken = isset($tokens['onedrive']['access_token']) ? $tokens['onedrive']['access_token'] : '';

                if (!empty($backup_OnedriveAccessToken)) {

	                if (!XAG_ONEDRIVE_KEY || !XAG_ONEDRIVE_SECRET) {
		                return array(
			                'status'  => 'error',
			                'message' => 'Please synchronize your Backup Settings to obtain latest updated settings.'
		                );
	                }

                    // Initialize the class
                    $onedriveClient = new OnedriveClient( array(
                        'client_id'         => XAG_ONEDRIVE_KEY,
                        'client_secret'     => XAG_ONEDRIVE_SECRET
                    ));

                    $onedriveClient->SetAccessToken($tokens['onedrive']);
                    $onedriveClient->renewAccessToken();

                    // Try to create a folder
                    $onedriveClient->downloadCall('/drive/root:/psv3/'.$backup.':/content', $backupFolder . '/' . $backup);
                } else {
                    return array(
                        'status'  => 'error',
                        'message' => 'Please synchronize your Backup Settings to obtain your OneDrive Access Token on '.$realSiteName.' website.'
                    );
                }
            }
            /* ONEDRIVE */

			/* GOOGLEDRIVE */
			if ($storage === 'googledrive') {
				// Include the DropBox Class
				$googleDriveClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'GoogleDriveClient.php';
				if ( ! class_exists( 'GoogleDriveClient' ) ) {
					require_once( $googleDriveClass );
				}

				// check the token
				$backup_GoogleDriveAccessToken = isset($tokens['googledrive']) ? $tokens['googledrive'] : '';

				if (!empty($backup_GoogleDriveAccessToken)) {


					if (!XAG_GOOGLEDRIVE_KEY || !XAG_GOOGLEDRIVE_SECRET) {
						return array(
							'status'  => 'error',
							'message' => 'Please synchronize your Backup Settings to obtain latest updated settings.'
						);
					}

					// Initialize the class
					$googleDriveClient = new GoogleDriveClient( array(
						'app_key'         => XAG_GOOGLEDRIVE_KEY,
						'app_secret'      => XAG_GOOGLEDRIVE_SECRET
					), $backup_GoogleDriveAccessToken );

					// Try to create a folder
					$googleDriveClient->Download($backup_id, $backupFolder . '/' . $backup);
				} else {
					return array(
						'status'  => 'error',
						'message' => 'Please synchronize your Backup Settings to obtain your Google Drive Access Token on '.$realSiteName.' website.'
					);
				}
			}
			/* GOOGLEDRIVE */

			/* AMAZON S3 */
			if ($storage === 'amazons3') {

				// Include the Amazon S3 Class
				$amazonClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'S3.php';
				if ( ! class_exists( 'S3' ) ) {
					require_once( $amazonClass );
				}

				$backup_AmazonAccessKey = isset( $tokens['amazon']['access_key'] ) ? $tokens['amazon']['access_key'] : '';
				$backup_AmazonSecretKey = isset( $tokens['amazon']['secret_key'] ) ? $tokens['amazon']['secret_key'] : '';
				$backup_AmazonBucket    = isset( $tokens['amazon']['bucket'] ) ? $tokens['amazon']['bucket'] : '';

				if (!empty($backup_AmazonAccessKey) && !empty($backup_AmazonSecretKey) && !empty($backup_AmazonBucket)) {
					S3::setAuth($backup_AmazonAccessKey, $backup_AmazonSecretKey);

					$backup_AmazonBucket = ltrim(rtrim($backup_AmazonBucket, '/'), '/');
					$backup_AmazonBucket = explode('/', $backup_AmazonBucket);

					if (($object = S3::getObject($backup_AmazonBucket[0], $backup_id, $backupFolder . '/' . $backup)) === false) {
						return array(
							'status'  => 'error',
							'message' => 'Failed to download the backup zip file from Amazon S3 on '.$realSiteName.' website.'
						);
					}

				} else {
					return array(
						'status'  => 'error',
						'message' => 'Please synchronize your Backup Settings to obtain your Amazon S3 credentials on '.$realSiteName.' website.'
					);
				}
			}
			/* AMAZON S3 */

			/* FTP */
			if ($storage === 'sftp') {
				$backup_FTP_Host     = isset( $tokens['sftp']['sftp_hostname'] )  ? $tokens['sftp']['sftp_hostname'] : '';
				$backup_FTP_Port     = isset( $tokens['sftp']['sftp_port'] )      ? $tokens['sftp']['sftp_port'] : '';
				$backup_FTP_Username = isset( $tokens['sftp']['sftp_username'] )  ? $tokens['sftp']['sftp_username'] : '';
				$backup_FTP_Password = isset( $tokens['sftp']['sftp_password'] )  ? $tokens['sftp']['sftp_password'] : '';
				$backup_FTP_Location = isset( $tokens['sftp']['sftp_directory'] ) ? $tokens['sftp']['sftp_directory'] : '';

				if (
					!empty($backup_FTP_Host) &&
					!empty($backup_FTP_Port) &&
					!empty($backup_FTP_Username) &&
					!empty($backup_FTP_Password) &&
					!empty($backup_FTP_Location)
				) {
					// set up basic connection
					$conn = @ftp_connect($backup_FTP_Host, $backup_FTP_Port);

					if ($conn === false) {
						return array(
							'status'   => 'error',
							'message'  => 'Failed to connect to your custom remote server on '.$realSiteName.'.'
						);
					}

					// login with username and password
					$log = @ftp_login($conn, $backup_FTP_Username, $backup_FTP_Password);

					if ($log === false) {
						return array(
							'status'   => 'error',
							'message'  => 'Failed to log in to your custom remote server on '.$realSiteName.'.'
						);
					}

					// download the file
					@ftp_get($conn, $backupFolder . '/' . $backup, $backup_FTP_Location . '/' . $siteName . '/' . $backup, FTP_BINARY);

					// close the connection
					@ftp_close($conn);
				}
			}
			/* FTP */

			// Check if backup was successful
			if (!file_exists($backupFolder . '/' . $backup)) {
				return array(
					'status'  => 'error',
					'message' => 'Failed to restore the backup! Unable to download it to '.$realSiteName.' website!'
				);
			}

			// unzip the backup to the root path
			/** Start the unZIP process */
			$archive = new PclZip($backupFolder . '/' . $backup);

			$unzip_result = $archive->extract(PCLZIP_OPT_PATH, ABSPATH, PCLZIP_OPT_REPLACE_NEWER);

			// if we fail to extract the zip, return the error
			if ($unzip_result == false) {
				return array(
					'status'  => 'error',
					'message' => 'Downloaded but failed to unzip the backup on '.$realSiteName.' website!'
				);
			}

			// remove the backup zip file
			@unlink($backupFolder . '/' . $backup);

			// most tricky part, restore MySQL
			$mysql_result = MXAG_Backups::restoreMySQL(ABSPATH . 'mysql-backup.sql');

			// remove the backup mysql file
			@unlink(ABSPATH . 'mysql-backup.sql');

			if ($mysql_result !== true) {
				return array(
					'status'  => 'error',
					'message' => 'Restored files backup but failed the restore the database backup with the following message(s) on website '.$realSiteName.': '.$mysql_result.'!'
				);
			}

			return array(
				'status'  => 'success',
				'message' => 'Successfully restored '.$backup.' on '.$realSiteName.'!'
			);
		}

		// Function that restores MySQL backup
		public static function restoreMySQL($location, $wait = false) {
			@set_time_limit ( 0 );
			@ini_set('memory_limit', '5120M');
			@ini_set("max_execution_time", "5000");
			@ini_set("max_input_time", "5000");

			$sql_query = @fread(@fopen($location, 'r'), @filesize($location));
			$sql_query = MXAG_Backups::remove_remarks($sql_query);

			$errors = array();

			$con = new mysqli(DB_HOST, DB_USER, DB_PASSWORD, DB_NAME);
			if ($con->connect_errno) {
				return 'Failed to establish MySQL connection.';
			}

			if ($wait) {

				$sql_query = MXAG_Backups::split_sql_file($sql_query, ";\n");
				foreach($sql_query as $sql){
					if (!$result = $con->query($sql)) {
						$errors[] = $con->error;
					}
				}

			} else {

				$result = $con->multi_query($sql_query);

			}

			$con->close();

			if (sizeof($errors) > 0) {
				return join("\n", $errors);
			}

			return true;
		}

		// Function that truncates backups based on a set limit
		public static function trimBackups() {
			$limit = get_option('ps_backup_limit');
			if (!$limit || empty($limit)) return;

			// Get the preferred backup location
			$backupLocation = get_option('ps_backup_location');

			// Get the needed AccessTokens
			$tokens = get_option('ps_backup_settings');

			// Domain folder name
			$folder = self::getName(site_url());

			/* DROPBOX */
			if ($backupLocation === 'dropbox') {
				// Include the DropBox Class
				$dropBoxClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'DropboxClient.php';
				if ( ! class_exists( 'DropboxClient' ) ) {
					require_once( $dropBoxClass );
				}

				// check the token
				$backup_DropboxAccessToken = isset($tokens['dropbox']['access_token']) ? $tokens['dropbox']['access_token'] : '';

				if (!empty($backup_DropboxAccessToken)) {

					if (!XAG_DROPBOX_KEY || !XAG_DROPBOX_SECRET) {
						return;
					}

					// Initialize the class
					$dropboxClient = new DropboxClient( array(
						'app_key'         => XAG_DROPBOX_KEY,
						'app_secret'      => XAG_DROPBOX_SECRET
					), $tokens['dropbox']['access_token'] );

					$files  = $dropboxClient->ListFolder($folder);
					if ($files !== false) {
						$output = array();
						foreach($files['entries'] as $file) {
							if ($file['.tag'] == 'file' && strpos($file['name'], $folder) !== false) {
								$output[] = array(
									'file' => $file['name'],
									'size' => $file['size'],
									'date' => $file['client_modified'],
									'id'   => $file['id']
								);
							}
						}
						usort($output, function($a, $b)
						{
							if ($a['date'] == $b['date']) {
								return 0;
							}
							return ($a['date'] > $b['date']) ? -1 : 1;
						});

						if (sizeof($output) > $limit) {
							$trimBy = sizeof($output) - $limit;
							$backupsForRemoval = array_slice($output, -$trimBy, $trimBy,true);

							// Remove backups
							foreach($backupsForRemoval as $backup) {
								$dropboxClient->Delete('/' . $folder . '/' . $backup['file']);
							}
						}
						return true;
					}
				}
			}
			/* DROPBOX */

			/* ONEDRIVE */
			if ($backupLocation === 'onedrive') {
				// Include the DropBox Class
				$oneDriveClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'OnedriveClient.php';
				if ( ! class_exists( 'OnedriveClient' ) ) {
					require_once( $oneDriveClass );
				}

				// check the token
				$backup_OnedriveAccessToken = isset($tokens['onedrive']['access_token']) ? $tokens['onedrive']['access_token'] : '';

				if (!empty($backup_OnedriveAccessToken)) {

					if (!XAG_ONEDRIVE_KEY || !XAG_ONEDRIVE_SECRET) {
						return;
					}

					// Initialize the class
					$onedriveClient = new OnedriveClient( array(
						'client_id'         => XAG_ONEDRIVE_KEY,
						'client_secret'     => XAG_ONEDRIVE_SECRET
					));

					$onedriveClient->SetAccessToken($tokens['onedrive']);
					$onedriveClient->renewAccessToken();

					$files  = $onedriveClient->GetFileFolder('/drive/root:/psv3:/children');

					// Find all backups
					if ( sizeof($files["value"]) !== 0 ) {

						$backups = $files["value"];
						$output = array();
						foreach ( $backups as $backup ) {
							$mystring = $backup["name"];
							$pos = strpos($mystring, $folder);
							if ($pos === false) {
								continue;
							} else {
								if ( $pos == 0 ) {
									$output[] = array(
										'file' => $backup["name"],
										'size' => $backup['size'],
										'date' => date('Y-m-d H:i:s', strtotime($backup["lastModifiedDateTime"])),
										'id'   => $backup['id']
									);
								} else {
									continue;
								}
							}
						}
						if ( sizeof($output) !== 0 ) {

							usort($output, function($a, $b)
							{
								if ($a['date'] == $b['date']) {
									return 0;
								}
								return ($a['date'] > $b['date']) ? -1 : 1;
							});

							if (sizeof($output) > $limit) {
								$trimBy = sizeof($output) - $limit;
								$backupsForRemoval = array_slice($output, -$trimBy, $trimBy,true);

								// Remove backups
								foreach($backupsForRemoval as $backup) {
									$onedriveClient->deleteCall('/drive/root:/psv3/' . $backup['file']);
								}
							}
							return true;
						}
					}
				}
			}
			/* ONEDRIVE */

			/* AMAZON S3 */
			if ($backupLocation === 'amazons3') {

				// Include the Amazon S3 Class
				$amazonClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'S3.php';
				if ( ! class_exists( 'S3' ) ) {
					require_once( $amazonClass );
				}

				$backup_AmazonAccessKey = isset( $tokens['amazon']['access_key'] ) ? $tokens['amazon']['access_key'] : '';
				$backup_AmazonSecretKey = isset( $tokens['amazon']['secret_key'] ) ? $tokens['amazon']['secret_key'] : '';
				$backup_AmazonBucket    = isset( $tokens['amazon']['bucket'] ) ? $tokens['amazon']['bucket'] : '';

				if (!empty($backup_AmazonAccessKey) && !empty($backup_AmazonSecretKey) && !empty($backup_AmazonBucket)) {
					S3::setAuth($backup_AmazonAccessKey, $backup_AmazonSecretKey);

					$backup_AmazonBucket = ltrim(rtrim($backup_AmazonBucket, '/'), '/');
					$backup_AmazonBucket = explode('/', $backup_AmazonBucket);

					$prefix = null;
					if (sizeof($backup_AmazonBucket) > 1) {
						$prefix = $backup_AmazonBucket[1];
					}

					if (($contents = S3::getBucket($backup_AmazonBucket[0], $prefix)) !== false) {

						$output = array();

						foreach($contents as $id=>$file) {
							if (strpos($file['name'], $folder) !== false) {
								$output[] = array(
									'file' => str_replace($folder . '/', '', $file['name']),
									'size' => $file['size'],
									'date' => date('Y-m-d H:i:s', $file['time']),
									'id'   => $id
								);
							}
						}

						usort($output, function($a, $b)
						{
							if ($a['date'] == $b['date']) {
								return 0;
							}
							return ($a['date'] > $b['date']) ? -1 : 1;
						});

						if (sizeof($output) > $limit) {
							$trimBy = sizeof($output) - $limit;
							$backupsForRemoval = array_slice($output, -$trimBy, $trimBy,true);

							// Remove backups
							foreach($backupsForRemoval as $backup) {
								S3::deleteObject($backup_AmazonBucket[0], $backup['id']);
							}
						}

					}

				}
			}
			/* AMAZON S3 */

			/* GOOGLEDRIVE */
			if ($backupLocation === 'googledrive') {
				// Include the DropBox Class
				$googleDriveClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'GoogleDriveClient.php';
				if ( ! class_exists( 'GoogleDriveClient' ) ) {
					require_once( $googleDriveClass );
				}

				// check the token
				$backup_GoogleDriveAccessToken = isset($tokens['googledrive']) ? $tokens['googledrive'] : '';

				if (!empty($backup_GoogleDriveAccessToken)) {

					if (!XAG_GOOGLEDRIVE_KEY || !XAG_GOOGLEDRIVE_SECRET) {
						return;
					}

					// Initialize the class
					$googleDriveClient = new GoogleDriveClient( array(
						'app_key'         => XAG_GOOGLEDRIVE_KEY,
						'app_secret'      => XAG_GOOGLEDRIVE_SECRET
					), $backup_GoogleDriveAccessToken );

					$files  = $googleDriveClient->ListFiles($folder);
					if ($files !== false) {

						$output = array();

						foreach($files['items'] as $file) {
							$output[] = array(
								'file' => $file['title'],
								'size' => $file['fileSize'],
								'date' => $file['createdDate'],
								'id'   => $file['id']
							);
						}
						usort($output, function($a, $b)
						{
							if ($a['date'] == $b['date']) {
								return 0;
							}
							return ($a['date'] > $b['date']) ? -1 : 1;
						});

						if (sizeof($output) > $limit) {
							$trimBy = sizeof($output) - $limit;
							$backupsForRemoval = array_slice($output, -$trimBy, $trimBy,true);

							// Remove backups
							foreach($backupsForRemoval as $backup) {
								$googleDriveClient->Delete($backup['id']);
							}
						}
						return true;
					}

				}
			}
			/* GOOGLEDRIVE */

			/* FTP */
			if ($backupLocation === 'sftp') {
				$backup_FTP_Host     = isset( $tokens['sftp']['sftp_hostname'] )  ? $tokens['sftp']['sftp_hostname'] : '';
				$backup_FTP_Port     = isset( $tokens['sftp']['sftp_port'] )      ? $tokens['sftp']['sftp_port'] : '';
				$backup_FTP_Username = isset( $tokens['sftp']['sftp_username'] )  ? $tokens['sftp']['sftp_username'] : '';
				$backup_FTP_Password = isset( $tokens['sftp']['sftp_password'] )  ? $tokens['sftp']['sftp_password'] : '';
				$backup_FTP_Location = isset( $tokens['sftp']['sftp_directory'] ) ? $tokens['sftp']['sftp_directory'] : '';

				if (
					!empty($backup_FTP_Host) &&
					!empty($backup_FTP_Port) &&
					!empty($backup_FTP_Username) &&
					!empty($backup_FTP_Password) &&
					!empty($backup_FTP_Location)
				) {
					// set up basic connection
					$conn = @ftp_connect($backup_FTP_Host, $backup_FTP_Port);

					if ($conn === false) {
						return false;
					}

					// login with username and password
					$log = @ftp_login($conn, $backup_FTP_Username, $backup_FTP_Password);

					if ($log === false) {
						return false;
					}

					// go to the specified directory
					$dir = @ftp_chdir($conn, $backup_FTP_Location);

					if ($dir == false) {
						return false;
					}

					// get all files
					$files = @ftp_nlist($conn, $backup_FTP_Location . '/' . $folder);

					$output = array();

					foreach($files as $file) {
						if ($file !== '.' && $file !== '..') {
							$output[] = array(
								'file' => $file,
								'size' => @ftp_size($conn, $backup_FTP_Location . '/' . $folder . '/' . $file),
								'date' => date('Y-m-d H:i:s', @ftp_mdtm($conn, $backup_FTP_Location . '/' . $folder . '/' . $file)),
								'id'   => $file
							);
						}
					}

					usort($output, function($a, $b)
					{
						if ($a['date'] == $b['date']) {
							return 0;
						}
						return ($a['date'] > $b['date']) ? -1 : 1;
					});


					if (sizeof($output) > $limit) {
						$trimBy = sizeof($output) - $limit;
						$backupsForRemoval = array_slice($output, -$trimBy, $trimBy,true);

						// Remove backups
						foreach($backupsForRemoval as $backup) {
							@ftp_delete($conn, $backup_FTP_Location . '/' . $folder . '/' . basename($backup['file']));
						}
					}

					// close the connection
					@ftp_close($conn);

					return true;
				}
			}
			/* FTP */
		}

		public static function doCloneBackup() {

			// Give PHP enough time to zip and upload a backup
			@set_time_limit(0);

			// Check whether the folders exist (if not, make them)
			$backupFolder = XAG_PATH . '/backups';

			if ( ! file_exists( $backupFolder ) ) {
				mkdir( $backupFolder );
			}

			// Remove the files backup (VERY IMPORTANT)
			$files = glob($backupFolder . '/*'); // get all file names
			foreach ( $files as $file ) { // iterate files
				if ( is_file($file) )
					unlink($file); // delete file
			}

			// Include the ZIP Class
			$zipClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'pclzip.lib.php';
			if (!class_exists('PclZip')) {
				require_once ( $zipClass );
			}

			// WordPress Directory
			$wordPressHomeDir = ABSPATH;

			$backupName = md5(time() . AUTH_KEY . AUTH_SALT);

			// Backup ZIP File
			$backupFile = $backupFolder . DIRECTORY_SEPARATOR . $backupName . '.zip';
			// Backup SQL File
			$backupSQL = $wordPressHomeDir . 'mysql-backup.sql';

			if ( file_exists( $backupSQL ) ) {
				unlink($backupSQL);
			}
			// Start the MySQL backup process
			self::doBackupMySQL($backupSQL);

			// Start the ZIP creation process
			$archive = new PclZip($backupFile);
			$listOfFilesAndFolders = array(
				$wordPressHomeDir . 'wp-admin' . DIRECTORY_SEPARATOR,
				$wordPressHomeDir . 'wp-content' . DIRECTORY_SEPARATOR,
				$wordPressHomeDir . 'wp-includes' . DIRECTORY_SEPARATOR,
				$wordPressHomeDir . 'index.php',
				$wordPressHomeDir . 'wp-activate.php',
				$wordPressHomeDir . 'wp-blog-header.php',
				$wordPressHomeDir . 'wp-comments-post.php',
				$wordPressHomeDir . 'wp-config.php',
				$wordPressHomeDir . 'wp-cron.php',
				$wordPressHomeDir . 'wp-links-opml.php',
				$wordPressHomeDir . 'wp-load.php',
				$wordPressHomeDir . 'wp-login.php',
				$wordPressHomeDir . 'wp-mail.php',
				$wordPressHomeDir . 'wp-settings.php',
				$wordPressHomeDir . 'wp-signup.php',
				$wordPressHomeDir . 'wp-trackback.php',
				$wordPressHomeDir . 'xmlrpc.php',
				$wordPressHomeDir . 'mysql-backup.sql',
			);

			// Zip the whole thing
			$status = $archive->create($listOfFilesAndFolders, PCLZIP_OPT_REMOVE_PATH, $wordPressHomeDir, PCLZIP_CB_PRE_ADD, 'xag_preZipAdd' );

			// Check if the zipping has been successful
			if ($status == 0) {
				return array(
					'status'  => 'error',
					'message' => 'Failed to create a clone of the website. Could not zip the files.'
				);
			}

			// Remote the MySQL backup from Root Dir (VERY IMPORTANT)
			unlink($backupSQL);

			return array(
				'status'  => 'success',
				'message' => 'Successfully created clone backup.',
				'data'    => content_url( DIRECTORY_SEPARATOR . 'plugins' . DIRECTORY_SEPARATOR . 'xagio' . DIRECTORY_SEPARATOR . 'backups' . DIRECTORY_SEPARATOR . basename($backupFile) )
			);
		}

		// Function that handles the overall backup process
		public static function doBackup() {

			try {

				$createID = '0';
				if ( isset($_POST['create_id']) ) $createID = $_POST['create_id'];

				// Get the preferred backup location
				$backupLocation = get_option('ps_backup_location');

				// Get the needed AccessTokens
				$tokens = get_option('ps_backup_settings');

				// Check if we have the tokens and location set
				if (empty($tokens) || empty($backupLocation) || $backupLocation == 'none') {
					if ( $createID === '0' ) {
						return array(
							'status'  => 'error',
							'message' => 'Please set your Storage Method and credentials for the preferred method.'
						);
					} else {
						MXAG_Api::apiRequest(
							$endpoint = 'backups',
							$method   = 'POST',
							array(
								'message' => 'Please set your Storage Method and credentials for the preferred method.',
								'create_id' => $createID
							),
							$http_code
						);
					}
				}

				// Give PHP enough time to zip and upload a backup
				@set_time_limit(0);

				// Check whether the folders exist (if not, make them)
				$backupFolder = XAG_PATH . '/backups';

				if ( ! file_exists( $backupFolder ) ) {
					mkdir( $backupFolder );
				}

				// Remote the files backup (VERY IMPORTANT)
				$files = glob($backupFolder . '/*'); // get all file names
				foreach ( $files as $file ) { // iterate files
					if ( is_file($file) )
						unlink($file); // delete file
				}

				// Include the ZIP Class
				$zipClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'pclzip.lib.php';
				if (!class_exists('PclZip')) {
					require_once ( $zipClass );
				}

				// WordPress Directory
				$wordPressHomeDir = ABSPATH;

				// Create Backup Name (both folder and file name)
				$siteName = self::getName(site_url());

				$backupName = $siteName . '_backup_' . date('m-d-Y_H-i');

				// Backup ZIP File
				$backupFile = $backupFolder . DIRECTORY_SEPARATOR . $backupName . '.zip';
				// Backup SQL File
				$backupSQL = $wordPressHomeDir . 'mysql-backup.sql';

				if ( file_exists( $backupSQL ) ) {
					unlink($backupSQL);
				}
				// Start the MySQL backup process
				self::doBackupMySQL($backupSQL);

				// Start the ZIP creation process
				$archive = new PclZip($backupFile);
				$listOfFilesAndFolders = array(
					$wordPressHomeDir . 'wp-admin' . DIRECTORY_SEPARATOR,
					$wordPressHomeDir . 'wp-content' . DIRECTORY_SEPARATOR,
					$wordPressHomeDir . 'wp-includes' . DIRECTORY_SEPARATOR,
					$wordPressHomeDir . 'index.php',
					$wordPressHomeDir . 'wp-activate.php',
					$wordPressHomeDir . 'wp-blog-header.php',
					$wordPressHomeDir . 'wp-comments-post.php',
					$wordPressHomeDir . 'wp-config.php',
					$wordPressHomeDir . 'wp-cron.php',
					$wordPressHomeDir . 'wp-links-opml.php',
					$wordPressHomeDir . 'wp-load.php',
					$wordPressHomeDir . 'wp-login.php',
					$wordPressHomeDir . 'wp-mail.php',
					$wordPressHomeDir . 'wp-settings.php',
					$wordPressHomeDir . 'wp-signup.php',
					$wordPressHomeDir . 'wp-trackback.php',
					$wordPressHomeDir . 'xmlrpc.php',
					$wordPressHomeDir . 'mysql-backup.sql',
				);

				// Zip the whole thing
				$status = $archive->create($listOfFilesAndFolders, PCLZIP_OPT_REMOVE_PATH, $wordPressHomeDir, PCLZIP_CB_PRE_ADD, 'xag_preZipAdd' );

				// Check if the zipping has been successful
				if ($status == 0) {
					if ( $createID === '0' ) {
						return array(
							'status'  => 'error',
							'message' => 'Failed to create a backup. Could not zip the files.'
						);
					} else {
						MXAG_Api::apiRequest(
							$endpoint = 'backups',
							$method   = 'POST',
							array(
								'message' => 'Failed to create a backup. Could not zip the files.',
								'create_id' => $createID
							),
							$http_code
						);
					}
				}

				// Remote the MySQL backup from Root Dir (VERY IMPORTANT)
				unlink($backupSQL);

				/**
				 *   Start uploading the backup to the remote server
				 */

				/* DROPBOX */
				if ($backupLocation === 'dropbox') {
					// Include the DropBox Class
					$dropBoxClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'DropboxClient.php';
					if ( ! class_exists( 'DropboxClient' ) ) {
						require_once( $dropBoxClass );
					}

					// check the token
					$backup_DropboxAccessToken = isset($tokens['dropbox']['access_token']) ? $tokens['dropbox']['access_token'] : '';

					if (!empty($backup_DropboxAccessToken)) {

						if (!XAG_DROPBOX_KEY || !XAG_DROPBOX_SECRET) {
							return array(
								'status'  => 'error',
								'message' => 'Please synchronize backup settings on this website before trying again.'
							);
						}

						// Initialize the class
						$dropboxClient = new DropboxClient( array(
							'app_key'         => XAG_DROPBOX_KEY,
							'app_secret'      => XAG_DROPBOX_SECRET
						), $backup_DropboxAccessToken );

						// Try to create a folder
						$dropboxClient->CreateFolder($siteName);

						// Upload the file
						$dropboxOutput = $dropboxClient->Upload($backupFile, '/' . $siteName);

						// Check if any errors
						if (isset($dropboxOutput['error_summary'])) {
							if ( $createID === '0' ) {
								return array(
									'status'  => 'error',
									'message' => 'Failed to upload backup to Dropbox. Info: ' . $dropboxOutput['error_summary']
								);
							} else {
								MXAG_Api::apiRequest(
									$endpoint = 'backups',
									$method   = 'POST',
									array(
										'message' => 'Failed to upload backup to Dropbox. Info: ' . $dropboxOutput['error_summary'],
										'create_id' => $createID
									),
									$http_code
								);
							}
						}

					}
				}
				/* DROPBOX */

				/* ONEDRIVE */
				if ($backupLocation === 'onedrive') {
					// Include the DropBox Class
					$oneDriveClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'OnedriveClient.php';
					if ( ! class_exists( 'OnedriveClient' ) ) {
						require_once( $oneDriveClass );
					}

					// check the token
					$backup_OnedriveAccessToken = isset($tokens['onedrive']) ? $tokens['onedrive'] : '';

					if (!empty($backup_OnedriveAccessToken)) {

						if (!XAG_ONEDRIVE_KEY || !XAG_ONEDRIVE_SECRET) {
							return array(
								'status'  => 'error',
								'message' => 'Please synchronize backup settings on this website before trying again.'
							);
						}

						// Initialize the class
						$onedriveClient = new OnedriveClient( array(
							'client_id'         => XAG_ONEDRIVE_KEY,
							'client_secret'     => XAG_ONEDRIVE_SECRET
						));

						$onedriveClient->SetAccessToken($backup_OnedriveAccessToken);
						$onedriveClient->renewAccessToken();

						// Try to create a folder
						$onedriveClient->CreateFolder();

						// Upload the file
						$onedriveOutput = $onedriveClient->upload($backupFile);

						// Check if any errors
						if (isset($onedriveOutput['error'])) {
							if ( $createID === '0' ) {
								return array(
									'status'  => 'error',
									'message' => 'Failed to upload backup to OneDrive. Info: ' . $onedriveOutput['error']['message']
								);
							} else {
								MXAG_Api::apiRequest(
									$endpoint = 'backups',
									$method   = 'POST',
									array(
										'message' => 'Failed to upload backup to OneDrive. Info: ' . $onedriveOutput['error']['message'],
										'create_id' => $createID
									),
									$http_code
								);
							}
						}
					}
				}
				/* ONEDRIVE */

				/* AMAZON S3 */
				if ($backupLocation === 'amazons3') {

					// Include the Amazon S3 Class
					$amazonClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'S3.php';
					if ( ! class_exists( 'S3' ) ) {
						require_once( $amazonClass );
					}

					$backup_AmazonAccessKey = isset( $tokens['amazon']['access_key'] ) ? $tokens['amazon']['access_key'] : '';
					$backup_AmazonSecretKey = isset( $tokens['amazon']['secret_key'] ) ? $tokens['amazon']['secret_key'] : '';
					$backup_AmazonBucket    = isset( $tokens['amazon']['bucket'] ) ? $tokens['amazon']['bucket'] : '';

					if (!empty($backup_AmazonAccessKey) && !empty($backup_AmazonSecretKey) && !empty($backup_AmazonBucket)) {
						S3::setAuth($backup_AmazonAccessKey, $backup_AmazonSecretKey);

						// Upload backup
						$amazonOutput = S3::putObject(S3::inputFile($backupFile, false), $backup_AmazonBucket, $siteName . '/' . basename($backupFile), S3::ACL_PUBLIC_READ);

						// Check if any errors
						if ($amazonOutput !== true) {
							if ( $createID === '0' ) {
								return array(
									'status'  => 'error',
									'message' => 'Failed to upload backup to Amazon S3. Info: ' . $amazonOutput
								);
							} else {
								MXAG_Api::apiRequest(
									$endpoint = 'backups',
									$method   = 'POST',
									array(
										'message' => 'Failed to upload backup to Amazon S3. Info: ' . $amazonOutput,
										'create_id' => $createID
									),
									$http_code
								);
							}
						}
					}
				}
				/* AMAZON S3 */

				/* GOOGLEDRIVE */
				if ($backupLocation === 'googledrive') {
					// Include the DropBox Class
					$googleDriveClass = XAG_PATH . DIRECTORY_SEPARATOR . 'ext' . DIRECTORY_SEPARATOR . 'GoogleDriveClient.php';
					if ( ! class_exists( 'GoogleDriveClient' ) ) {
						require_once( $googleDriveClass );
					}

					// check the token
					$backup_GoogleDriveAccessToken = isset($tokens['googledrive']) ? $tokens['googledrive'] : '';

					if (!empty($backup_GoogleDriveAccessToken)) {

						if (!XAG_GOOGLEDRIVE_KEY || !XAG_GOOGLEDRIVE_SECRET) {
							return array(
								'status'  => 'error',
								'message' => 'Please synchronize backup settings on this website before trying again.'
							);
						}

						// Initialize the class
						$googleDriveClient = new GoogleDriveClient( array(
							'app_key'         => XAG_GOOGLEDRIVE_KEY,
							'app_secret'      => XAG_GOOGLEDRIVE_SECRET
						), $backup_GoogleDriveAccessToken );

						// Try to create a root PSv3 folder
						$googleDriveClient->CreateFolder('ProjectSupremacyV3');

						// Try to create a sitename folder
						$googleDriveClient->CreateFolder( $siteName, 'ProjectSupremacyV3');

						// Upload the file
						$googleDriveOutput = $googleDriveClient->Upload($backupFile, $siteName);

						// Check if any errors
						if (isset($googleDriveOutput['error'])) {
							if ( $createID === '0' ) {
								return array(
									'status'  => 'error',
									'message' => 'Failed to upload backup to Google Drive. Info: ' . $googleDriveOutput['errors'][0]['message']
								);
							} else {
								MXAG_Api::apiRequest(
									$endpoint = 'backups',
									$method   = 'POST',
									array(
										'message' => 'Failed to upload backup to Google Drive. Info: ' . $googleDriveOutput['errors'][0]['message'],
										'create_id' => $createID
									),
									$http_code
								);
							}
						}
					}
				}
				/* GOOGLEDRIVE */

				/* FTP */
				if ($backupLocation === 'sftp') {
					$backup_FTP_Host     = isset( $tokens['sftp']['sftp_hostname'] )  ? $tokens['sftp']['sftp_hostname'] : '';
					$backup_FTP_Port     = isset( $tokens['sftp']['sftp_port'] )      ? $tokens['sftp']['sftp_port'] : '';
					$backup_FTP_Username = isset( $tokens['sftp']['sftp_username'] )  ? $tokens['sftp']['sftp_username'] : '';
					$backup_FTP_Password = isset( $tokens['sftp']['sftp_password'] )  ? $tokens['sftp']['sftp_password'] : '';
					$backup_FTP_Location = isset( $tokens['sftp']['sftp_directory'] ) ? $tokens['sftp']['sftp_directory'] : '';

					if (
						!empty($backup_FTP_Host) &&
						!empty($backup_FTP_Port) &&
						!empty($backup_FTP_Username) &&
						!empty($backup_FTP_Password) &&
						!empty($backup_FTP_Location)
					) {
						// set up basic connection
						$conn = @ftp_connect($backup_FTP_Host, $backup_FTP_Port);

						if ($conn === false) {
							if ( $createID === '0' ) {
								return array(
									'status'   => 'error',
									'message'  => 'Failed to upload backup to SFTP. Info: Failed to log in to your custom remote server.'
								);
							} else {
								MXAG_Api::apiRequest(
									$endpoint = 'backups',
									$method   = 'POST',
									array(
										'message'  => 'Failed to upload backup to SFTP. Info: Failed to log in to your custom remote server.',
										'create_id' => $createID
									),
									$http_code
								);
							}
						}

						// login with username and password
						$log = @ftp_login($conn, $backup_FTP_Username, $backup_FTP_Password);

						if ($log === false) {
							if ( $createID === '0' ) {
								return array(
									'status'   => 'error',
									'message'  => 'Failed to upload backup to SFTP. Info: Failed to log in to your custom remote server.'
								);
							} else {
								MXAG_Api::apiRequest(
									$endpoint = 'backups',
									$method   = 'POST',
									array(
										'message'  => 'Failed to upload backup to SFTP. Info: Failed to log in to your custom remote server.',
										'create_id' => $createID
									),
									$http_code
								);
							}
						}

						// go to the specified directory
						$dir = @ftp_chdir($conn, $backup_FTP_Location);

						if ($dir == false) {
							if ( $createID === '0' ) {
								return array(
									'status'   => 'error',
									'message'  => 'Failed to upload backup to SFTP. Info: Failed to log in to your custom remote server.'
								);
							} else {
								MXAG_Api::apiRequest(
									$endpoint = 'backups',
									$method   = 'POST',
									array(
										'message'  => 'Failed to upload backup to SFTP. Info: Failed to log in to your custom remote server.',
										'create_id' => $createID
									),
									$http_code
								);
							}
						}

						// try to create a sitename folder
						@ftp_mkdir($conn, $backup_FTP_Location . '/' . $siteName);

						// upload a file
						if (
						!@ftp_put($conn, $backup_FTP_Location . '/' . $siteName . '/' . basename($backupFile), $backupFile, FTP_BINARY)
						) {
							if ( $createID === '0' ) {
								return array(
									'status'   => 'error',
									'message'  => 'Failed to upload backup to SFTP. Info: Failed to log in to your custom remote server.'
								);
							} else {
								MXAG_Api::apiRequest(
									$endpoint = 'backups',
									$method   = 'POST',
									array(
										'message'  => 'Failed to upload backup to SFTP. Info: Failed to log in to your custom remote server.',
										'create_id' => $createID
									),
									$http_code
								);
							}
						}

						// close the connection
						@ftp_close($conn);
					}
				}
				/* FTP */

				// Remote the files backup (VERY IMPORTANT)
				unlink($backupFile);

				// Trim backups
				self::trimBackups();

				if ( $createID === '0' ) {
					return array(
						'status'  => 'success',
						'message' => 'Backup successfully created.'
					);
				} else {
					MXAG_Api::apiRequest(
						$endpoint = 'backups',
						$method   = 'POST',
						array(
							'message' => 'Backup successfully created.',
							'create_id' => $createID
						),
						$http_code
					);
				}

			} catch (Exception $error) {
				XAG_Init::json('error', 'Plugin error occured. Please pass this down to support: ' . $error->getMessage());
			}

		}

		// Function that handle MySQL backup
		public static function doBackupMySQL( $location ) {

			global $wpdb;

			$return = '';

			$tables = array();
			$result = $wpdb->get_results( 'SHOW TABLES', ARRAY_N );

			foreach ( $result as $row ) {
				$tables[] = $row[0];
			}

			//cycle through
			foreach ( $tables as $table ) {
				$result     = $wpdb->get_results( 'SELECT * FROM ' . $table, ARRAY_A );
				$num_fields = sizeof( $result );

				$return .= 'DROP TABLE IF EXISTS ' . $table . ';';
				$row2 = $wpdb->get_results( 'SHOW CREATE TABLE ' . $table, ARRAY_N );

				$return .= "\n" . $row2[0][1] . ";\n";

				foreach ( $result as $row ) {
					$return .= 'INSERT INTO ' . $table . ' VALUES(';
					$data = array();
					foreach($row as $key=>$value) {
						$data[] = '"' . esc_sql($value) . '"';
					}
					$return .= join(',', $data);
					$return .= ");\n";
				}
			}

			/* Prevention for replaced % char from placeholder string */
			$percntSymbol = "%";
			$percntSymbolStr = esc_sql($percntSymbol);
			$return = str_replace($percntSymbolStr, '&#37;', $return);

			// Write the file
			file_put_contents( $location, $return );
		}

		// Functions that handle SQL files
		public static function remove_comments(&$output)
		{
			$lines = explode("\n", $output);
			$output = "";

			// try to keep mem. use down
			$linecount = count($lines);

			$in_comment = false;
			for($i = 0; $i < $linecount; $i++)
			{
				if( preg_match("/^\/\*/", preg_quote($lines[$i])) )
				{
					$in_comment = true;
				}

				if( !$in_comment )
				{
					$output .= $lines[$i] . "\n";
				}

				if( preg_match("/\*\/$/", preg_quote($lines[$i])) )
				{
					$in_comment = false;
				}
			}

			unset($lines);
			return $output;
		}
		public static function remove_remarks($sql)
		{
			$lines = explode("\n", $sql);

			// try to keep mem. use down
			$sql = "";

			$linecount = count($lines);
			$output = "";

			for ($i = 0; $i < $linecount; $i++)
			{
				if (($i != ($linecount - 1)) || (strlen($lines[$i]) > 0))
				{
					if (isset($lines[$i][0]) && $lines[$i][0] != "#")
					{
						$output .= $lines[$i] . "\n";
					}
					else
					{
						$output .= "\n";
					}
					// Trading a bit of speed for lower mem. use here.
					$lines[$i] = "";
				}
			}

			return $output;

		}
		public static function split_sql_file($sql, $delimiter)
		{
			// Split up our string into "possible" SQL statements.
			$tokens = explode($delimiter, $sql);

			// try to save mem.
			$sql = "";
			$output = array();

			// we don't actually care about the matches preg gives us.
			$matches = array();

			// this is faster than calling count($oktens) every time thru the loop.
			$token_count = count($tokens);
			for ($i = 0; $i < $token_count; $i++)
			{
				// Don't wanna add an empty string as the last thing in the array.
				if (($i != ($token_count - 1)) || (strlen($tokens[$i] > 0)))
				{
					// This is the total number of single quotes in the token.
					$total_quotes = preg_match_all("/'/", $tokens[$i], $matches);
					// Counts single quotes that are preceded by an odd number of backslashes,
					// which means they're escaped quotes.
					$escaped_quotes = preg_match_all("/(?<!\\\\)(\\\\\\\\)*\\\\'/", $tokens[$i], $matches);

					$unescaped_quotes = $total_quotes - $escaped_quotes;

					// If the number of unescaped quotes is even, then the delimiter did NOT occur inside a string literal.
					if (($unescaped_quotes % 2) == 0)
					{
						// It's a complete sql statement.
						$output[] = $tokens[$i];
						// save memory.
						$tokens[$i] = "";
					}
					else
					{
						// incomplete sql statement. keep adding tokens until we have a complete one.
						// $temp will hold what we have so far.
						$temp = $tokens[$i] . $delimiter;
						// save memory..
						$tokens[$i] = "";

						// Do we have a complete statement yet?
						$complete_stmt = false;

						for ($j = $i + 1; (!$complete_stmt && ($j < $token_count)); $j++)
						{
							// This is the total number of single quotes in the token.
							$total_quotes = preg_match_all("/'/", $tokens[$j], $matches);
							// Counts single quotes that are preceded by an odd number of backslashes,
							// which means they're escaped quotes.
							$escaped_quotes = preg_match_all("/(?<!\\\\)(\\\\\\\\)*\\\\'/", $tokens[$j], $matches);

							$unescaped_quotes = $total_quotes - $escaped_quotes;

							if (($unescaped_quotes % 2) == 1)
							{
								// odd number of unescaped quotes. In combination with the previous incomplete
								// statement(s), we now have a complete statement. (2 odds always make an even)
								$output[] = $temp . $tokens[$j];

								// save memory.
								$tokens[$j] = "";
								$temp = "";

								// exit the loop.
								$complete_stmt = true;
								// make sure the outer loop continues at the right point.
								$i = $j;
							}
							else
							{
								// even number of unescaped quotes. We still don't have a complete statement.
								// (1 odd and 1 even always make an odd)
								$temp .= $tokens[$j] . $delimiter;
								// save memory.
								$tokens[$j] = "";
							}

						} // for..
					} // else
				}
			}

			return $output;
		}

        public static function getName($url, $only_host = false){
            $parts = parse_url($url);
            if ($only_host) {
            	$name = $parts['host'];
            } else {
	            $name = str_replace(".", "_", $parts['host']);
            }
            return $name;
        }

        private static function loadKeys() {

	        $backup_settings = get_option('ps_backup_settings');
	        $secret_keys     = @$backup_settings['secret_keys'];

	        /** Backup Application Keys */
	        define('XAG_DROPBOX_KEY'       , isset($secret_keys['dropbox']) ? $secret_keys['dropbox']['public'] : false);
	        define('XAG_DROPBOX_SECRET'    , isset($secret_keys['dropbox']) ? $secret_keys['dropbox']['private'] : false);
	        define('XAG_ONEDRIVE_KEY'       , isset($secret_keys['onedrive']) ? $secret_keys['onedrive']['public'] : false);
	        define('XAG_ONEDRIVE_SECRET'    , isset($secret_keys['onedrive']) ? $secret_keys['onedrive']['private'] : false);
	        define('XAG_GOOGLEDRIVE_KEY'   , isset($secret_keys['googledrive']) ? $secret_keys['googledrive']['public'] : false);
	        define('XAG_GOOGLEDRIVE_SECRET', isset($secret_keys['googledrive']) ? $secret_keys['googledrive']['private'] : false);
	        /** Backup Application Keys */

        }

	}

}
