<?php

if ( ! class_exists( 'MXAG_Schema' ) ) {

	class MXAG_Schema {

		public static function initialize() {
			// Disable Model if not license is present
			if (!XAG_Licencator::isLicenseSet()) return;

			add_action( 'wp_head', array( 'MXAG_Schema', 'generateSchema' ) );
			add_action( 'admin_post_xag_validate_schema', array( 'MXAG_Schema', 'validateSchema' ) );
			add_action( 'admin_post_xag_render_schema', array( 'MXAG_Schema', 'renderSchema' ) );
			add_action( 'admin_post_xag_get_remote_schema', array( 'MXAG_Schema', 'getRemoteSchema' ) );
			add_action( 'admin_post_xag_get_remote_schema_groups', array( 'MXAG_Schema', 'getRemoteSchemaGroups' ) );
			add_action( 'admin_post_xag_save_schema', array( 'MXAG_Schema', 'saveSchema' ) );
			add_action( 'admin_post_xag_schema_wizard', array( 'MXAG_Schema', 'schemaWizard' ) );
		}

		public static function schemaWizard() {
			$post_id = $_POST['post_id'];
			$name    = $_POST['name'];
			$schema  = $_POST['swFields'];

			// Finish the schema
			$schema["@context"] = "http://schema.org";
			$schema["@type"]    = $_POST['type'];

			MXAG_Api::apiRequest( 'schema_wizard', 'POST', array(
					'domain'  => preg_replace( '/^www\./', '', $_SERVER['SERVER_NAME'] ),
					'schema'  => serialize( $schema ),
					'name'    => $name,
					'post_id' => $post_id
				) );

			XAG_Init::json( 'success', 'Schema has been generated and assigned to this page/post. Refreshing this window in order for you to see the changes.', $schema );
		}

		public static function renderSchema() {
			$post_id = $_POST['id'];
			$schema  = get_post_meta( $post_id, 'ps_schema_data', true );

			if ( $schema === false || empty( $schema ) ) {
				XAG_Init::json( 'error', 'Schema is not assigned for this page/post. Please save your page/post changes and try again.' );
			} else {
				XAG_Init::json( 'success', 'Schema rendered.', $schema );
			}
		}

		public static function validateSchema() {
			if ( isset( $_POST['url'] ) ) {
				$URL = trim( $_POST['url'] );

				$structuredData_URL = 'https://search.google.com/structured-data/testing-tool/validate';

				$postdata = self::buildQuery( array(
						'url' => $URL,
					) );

				if ( ini_get( 'allow_url_fopen' ) ) {
					$opts = array(
						'http' => array(
							'method'  => 'POST',
							'header'  => 'Content-type: application/x-www-form-urlencoded',
							'content' => $postdata
						)
					);

					$context = stream_context_create( $opts );
					$result  = file_get_contents( $structuredData_URL, false, $context );
				} else if ( function_exists( 'curl_init' ) ) {
					$ch = curl_init();

					curl_setopt( $ch, CURLOPT_URL, $structuredData_URL );
					curl_setopt( $ch, CURLOPT_POST, true );
					curl_setopt( $ch, CURLOPT_POSTFIELDS, $postdata );
					curl_setopt( $ch, CURLOPT_HTTPHEADER, array( 'Content-Type: application/x-www-form-urlencoded' ) );

					curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );

					$result = curl_exec( $ch );

					curl_close( $ch );
				} else {
					XAG_Init::json( 'error', 'Your server does not have cURL installed nor does it support usage of *allow_url_fopen* in its PHP configuration. Please fix one of these to get the Schema Validation working.' );
				}


				$result = str_replace( ")]}'", '', $result );
				$result = str_replace( "\n", '', $result );
				$result = json_decode( $result, true );

				if ( ! $result ) {
					XAG_Init::json( 'error', 'Failed to decode JSON response.' );
				} else {
					XAG_Init::json( 'success', 'Done.', $result );
				}

			} else {
				XAG_Init::json( 'error', 'URL is missing from your query.' );
			}
		}

		public static function getRemoteRenderedSchemas( $ids = array(), $page_id = null, $type = 'post', &$output = null ) {
			$license_email = '';
			$license_key   = '';
			if ( ! $license_set = XAG_Licencator::isLicenseSet( $license_email, $license_key ) ) {
				return false;
			}

			// Set the domain name
			$domain = preg_replace( '/^www\./', '', $_SERVER['SERVER_NAME'] );

			// Set the HTTP Query
			$http_query = array(
				'license_email' => $license_email,
				'license_key'   => $license_key,
				'schema_id'     => join( ',', $ids ),
				'domain'        => $domain
			);

			if ( $page_id !== null ) {
				$http_query['page_id'] = $page_id;
				$http_query['type']    = $type;
			}

			$response = wp_remote_post( XAG_PANEL_URL . "/api/schema", array(
					'user-agent'  => "PSv3 - " . XAG_CURRENT_VERSION . " ($domain)",
					'timeout'     => 30,
					'redirection' => 5,
					'httpversion' => '1.0',
					'blocking'    => true,
					'method'      => 'POST',
					'body'        => $http_query,
				) );

			$output = $response;

			if ( is_wp_error( $response ) ) {
				return false;
			} else {
				if ( ! isset( $response['body'] ) ) {
					return false;
				} else {
					$data = json_decode( $response['body'], true );
					if ( ! $data ) {
						return false;
					} else {
						return $data;
					}
				}
			}
		}

		public static function getRemoteSchemaGroups() {
			$license_email = '';
			$license_key   = '';
			if ( ! $license_set = XAG_Licencator::isLicenseSet( $license_email, $license_key ) ) {
				XAG_Init::json( 'error', 'Your license is invalid. Please go to Panel and troubleshoot this issue.' );
			}

			// Set the domain name
			$domain = preg_replace( '/^www\./', '', $_SERVER['SERVER_NAME'] );

			// Set the HTTP Query
			$http_query = array(
				'license_email' => $license_email,
				'license_key'   => $license_key
			);

			// Build HTTP Query
			$http_query = self::buildQuery( $http_query );

			$response = wp_remote_get( XAG_PANEL_URL . "/api/schema_groups?$http_query", array(
					'user-agent'  => "PSv3 - " . XAG_CURRENT_VERSION . " ($domain)",
					'timeout'     => 30,
					'redirection' => 5,
					'httpversion' => '1.0',
					'blocking'    => true
				) );

			if ( is_wp_error( $response ) ) {
				XAG_Init::json( 'error', 'The license information that you submitted is not valid. Please try again.' );
			} else {
				if ( ! isset( $response['body'] ) ) {
					XAG_Init::json( 'error', 'We are experiencing temporary problems with our servers. Please try again later.' );
				} else {
					$data = json_decode( $response['body'], true );
					if ( ! $data ) {
						XAG_Init::json( 'error', 'Failed to decode JSON response!' );
					} else {
						header( 'Content-Type: application/json' );
						echo json_encode( $data );
					}
				}
			}
		}

		public static function getRemoteSchema() {
			$license_email = '';
			$license_key   = '';
			if ( ! $license_set = XAG_Licencator::isLicenseSet( $license_email, $license_key ) ) {
				XAG_Init::json( 'error', 'Your license is invalid. Please go to Panel and troubleshoot this issue.' );
			}

			// Set the domain name
			$domain = preg_replace( '/^www\./', '', $_SERVER['SERVER_NAME'] );

			// Set the HTTP Query
			$http_query = array(
				'license_email' => $license_email,
				'license_key'   => $license_key
			);

			if ( isset( $_POST['schema_group'] ) ) {
				$http_query['schema_group'] = $_POST['schema_group'];
			}

			// Build HTTP Query
			$http_query = self::buildQuery( $http_query );

			$response = wp_remote_get( XAG_PANEL_URL . "/api/schema?$http_query", array(
					'user-agent'  => "PSv3 - " . XAG_CURRENT_VERSION . " ($domain)",
					'timeout'     => 30,
					'redirection' => 5,
					'httpversion' => '1.0',
					'blocking'    => true
				) );

			if ( is_wp_error( $response ) ) {
				XAG_Init::json( 'error', 'The license information that you submitted is not valid. Please try again.' );
			} else {
				if ( ! isset( $response['body'] ) ) {
					XAG_Init::json( 'error', 'We are experiencing temporary problems with our servers. Please try again later.' );
				} else {

					$data = json_decode( $response['body'], true );
					if ( ! $data ) {
						XAG_Init::json( 'error', 'Failed to decode JSON response!' );
					} else {
						header( 'Content-Type: application/json' );
						echo json_encode( $data );
					}
				}
			}
		}

		public static function buildQuery( $params ) {
			return http_build_query( $params, '', '&' );
		}

		public static function getSchemas( $post_id = null, $type = 'post', $meta = false ) {
			if ($type == 'post') {
				if ( $post_id == null ) {
					$schemas = get_option( 'ps_schema_meta' );
				} else {
					if ( $post_id == get_option( 'page_on_front' ) ) {
						$schemas = get_option( 'ps_schema_meta' );
					} else {
						$schemas = get_post_meta( $post_id, 'ps_schema_meta', true );
					}
				}
			} else if ($type == 'term') {

				if (!isset($meta['ps_schema_meta'])) {
					$schemas = false;
				} else if (empty($meta['ps_schema_meta'])) {
					$schemas = false;
				} else {
					$schemas = $meta['ps_schema_meta'];
				}

			}
			if ( ! $schemas ) {
				return false;
			} else {
				return $schemas;
			}
		}

		public static function applyShortcodes( $schemas ) {
			$new_schemas = array();
			foreach ( $schemas as $schema ) {
				$new_schemas[] = self::_applyRecursive( $schema );
			}

			return $new_schemas;
		}

		private static function _applyRecursive( $schema = array() ) {
			global $post;
			$ID = null;
			if ( MXAG_Seo::is_home_posts_page() || MXAG_Seo::is_posts_page() || MXAG_Seo::is_home_static_page() ) {
				$ID = 0;
			} else {
				$ID = $post->ID;
			}

			$new_schema = array();
			foreach ( $schema as $key => $value ) {
				if ( is_array( $value ) ) {
					unset( $schema[ $key ] );
					$new_schema[ $key ] = self::_applyRecursive( $value );
				} else {
					// Replace :amp; for & as the & would split into different vars.
					$new_schema[ $key ] = do_shortcode( MXAG_Seo::replaceVars( $value, $ID ) );
					unset( $schema[ $key ] );
				}
			}

			return $new_schema;
		}

		public static function applyReviews($reviews = array(), $natural_reviews = false, &$schemas) {

			// Check if feature is enabled
			$ps_features = get_option('ps_features');
			if ($ps_features != false && is_array($ps_features)) {
				if (!in_array('reviews', $ps_features)) {
					return;
				}
			}
			if ($ps_features == 'none') return;

			// Loop through all Schemas and inject Ratings if there are reviews
			if ( sizeof( $reviews ) > 0 && is_array( $reviews ) ) {

				// Calculate how much reviews / ratings and their sum value
				$reviewCount = 0;
				$ratingCount = 0;

				// Store all values
				$ratingValues = array();

				// Array to hold all reviews
				$reviewSchemas = array();

				// Loop through all reviews / ratings
				foreach ( $reviews as $review ) {

					$ratingValues[] = $review['rating'];

					// Fix for old versions
					if (!isset($review['stars_only'])) {
						$review['stars_only'] = 0;
					}

					if ($review['stars_only'] == 1) {

						$ratingCount++;

					} else {

						$reviewCount++;

						$reviewSchemas[] = array(
							'@type'         => 'Review',
							'author'        => $review['name'],
							'datePublished' => $review['date'],
							'description'   => $review['review'],
							'reviewRating'  => array(
								'@type'       => 'Rating',
								'ratingValue' => $review['rating']
							)
						);

					}
				}

				for ( $i = 0; $i < sizeof( $schemas ); $i ++ ) {

					// If current schema supports reviews
					if ( in_array( $schemas[ $i ]['@type'], MXAG_Reviews::allowedReviewSchemas() ) && sizeof($reviewSchemas) > 0 ) {

						$schemas[ $i ]['review'] = $reviewSchemas;

					}

					// If reviews should be injected into AggregateRating Schemas
					if ( $natural_reviews == true ) {

						$aggregateRating;

						if ( $schemas[ $i ]['@type'] == 'AggregateRating' ) {
							$aggregateRating = &$schemas[ $i ];
						} else if ( isset( $schemas[ $i ]['aggregateRating'] ) ) {
							$aggregateRating = &$schemas[ $i ]['aggregateRating'];
						}

						if ( ($reviewCount > 0 || $ratingCount > 0) && sizeof($ratingValues) > 0 ) {

							// I HATE THIS, but it has to be here
							if ($reviewCount > 0) {
								$aggregateRating['reviewCount'] = $reviewCount;
							} else {
								unset($aggregateRating['reviewCount']);
							}
							if ($ratingCount > 0) {
								$aggregateRating['ratingCount'] = $ratingCount;
							} else {
								unset($aggregateRating['ratingCount']);
							}

							if (!isset($aggregateRating['bestRating'])) {
								$aggregateRating['bestRating'] = 5;
							}

							if (empty($aggregateRating['worstRating'])) {
								$aggregateRating['worstRating'] = 1;
							}

							// Current Rating Value
							$ratingValue = 0;

							// Best Rating Calculation
							$bestRating  = $aggregateRating['bestRating'];
							$worstRating = $aggregateRating['worstRating'];

							// Check for Reviews
							if (isset($schemas[ $i ]['review'])) {
								if (is_array($schemas[ $i ]['review'])) {
									for($r = 0; $r < sizeof($schemas[ $i ]['review'][ $r ]); $r++) {

										$tempRating = $schemas[ $i ]['review'][ $r ]['reviewRating']['ratingValue'];
										$starTemp   = (($bestRating - $worstRating) / 4);
										$tempSum    = ($starTemp * $tempRating) + ($worstRating - $starTemp);

										// I don't know why I even try
										$schemas[ $i ]['review'][ $r ]['reviewRating']['worstRating'] = $worstRating;
										$schemas[ $i ]['review'][ $r ]['reviewRating']['bestRating']  = $bestRating;
										$schemas[ $i ]['review'][ $r ]['reviewRating']['ratingValue'] = number_format($tempSum, 2, '.', '');
									}
								}
							}

							// Calculate the ratings
							foreach($ratingValues as $tempRatingValue) {

								$starTemp    = (($bestRating - $worstRating) / 4);
								$tempSum     = ($starTemp * $tempRatingValue) + ($worstRating - $starTemp);
								$ratingValue += $tempSum;

							}

							// Total rating sum
							$ratingValue = $ratingValue / ($reviewCount + $ratingCount);

							// Format number
							$ratingValue = number_format($ratingValue, 2, '.', '');

							// Set the temp rating value
							$GLOBALS['currentRatingValue'] = $ratingValue;

							// Set the Rating Value
							$aggregateRating['ratingValue'] = $ratingValue;
						}

					}
				}
			}

		}

		public static function generateSchema() {
			global $post;

			if(!empty($post->ID)){

				$review_page_id = $post->ID;

				// Review Settings
				$ps_review = get_option( 'ps_review' );

				// Homepage Schema
				if ( MXAG_Seo::is_home_posts_page() || MXAG_Seo::is_posts_page() || MXAG_Seo::is_home_static_page() ) {
					$schemas = get_option( 'ps_schema_data' );

				// Taxonomy Schema
				} else if ( is_category() || is_tag() || is_tax()  ) {

					$schemas = false;

					$object = $GLOBALS['wp_query']->get_queried_object();
					if ( is_object( $object ) ) {

						$taxonomy = $object->taxonomy;
						$meta     = get_option( $taxonomy . '_' . $object->term_id );

						if (isset($meta['ps_schema_data'])) {
							if (!empty($meta['ps_schema_data'])) {
								$schemas = $meta['ps_schema_data'];
							}
						}

					}

					// Use global taxonomy schema
					if ($schemas == false) {
						// TODO --- need to add global taxonomies
					}

				// Post Schema
				} else {
					$schemas = get_post_meta( $post->ID, 'ps_schema_data', true );

					// See if we have XAG_FORCE_SCHEMA activated
					if ( $schemas == false && XAG_FORCE_SCHEMA == true) {
						$schemas = get_option( 'ps_schema_data' );
					}
				}
				if ( $schemas != false ) {

					// Get all Reviews
					if ( @$ps_review['settings']['per_page_reviews'] == 1 ) {
						$reviews = MXAG_Reviews::getReviewsForPage( $review_page_id, null );
					} else {
						$reviews = MXAG_Reviews::getReviewsGlobal(null);
					}

					// Apply the reviews
					self::applyReviews($reviews, @$ps_review['settings']['natural_reviews'] == 1, $schemas);

					// Apply shortcodes and templates to schemas
					$schemas = self::applyShortcodes( $schemas );

					if ( is_array( $schemas ) && sizeof( $schemas ) == 1 ) {
						$schemas = $schemas[0];
					}

					// Check if we should render pretty schemas
					$PRETTY_SCHEMAS = 0;
					if (XAG_PRETTY_SCHEMA == true) {
						$PRETTY_SCHEMAS = 128;
					}

					$generatedSchema = json_encode( $schemas, $PRETTY_SCHEMAS );

					// Replace \\ with \
					$generatedSchema = str_replace( "\\\\", "", $generatedSchema );

					if ( XAG_REMOVE_FOOTPRINT == false ) {
						echo "\n\n<!-- Xagio – Schema.org -->\n";
					}

					echo '<script type="application/ld+json">' . "\n";
					echo $generatedSchema . "\n";
					echo "</script>";

					if ( XAG_REMOVE_FOOTPRINT == false ) {
						echo "\n<!-- Xagio – Schema.org -->\n\n";
					}

				}

			}			
		}
	}

}