<?php

if (!class_exists('MXAG_Seo')) {

	class MXAG_Seo
	{

		public static function initialize()
		{
			// Add Default Settings for Dynamic options
			if (is_admin()) {
				wp_schedule_single_event(time() + 5, 'xag_set_default_post_settings');
				wp_schedule_single_event(time() + 5, 'xag_set_default_taxonomy_settings');
			}

			// Disable Model if not license is present
			if (!XAG_Licencator::isLicenseSet()) {
				return;
			}

			// Check if feature is enabled
			$ps_features = get_option('ps_features');
			if ($ps_features != FALSE && is_array($ps_features)) {
				if (!in_array('seo', $ps_features)) {
					return;
				}
			}
			if ($ps_features == 'none') {
				return;
			}

			// Default Settings
			add_action('xag_set_default_post_settings', ['MXAG_Seo', 'setDefaultPostSettings']);
			add_action('xag_set_default_taxonomy_settings', ['MXAG_Seo', 'setDefaultTaxonomySettings']);

			// Save Post
			add_action('save_post', ['MXAG_Seo', 'save']);

			// Save settings
			add_action('admin_post_xag_save_general', ['MXAG_Seo', 'saveGeneral']);
			add_action('admin_post_xag_save_posttypes', ['MXAG_Seo', 'savePostTypes']);
			add_action('admin_post_xag_save_customposttypesOG', ['MXAG_Seo', 'saveCustomPostTypesOG']);
			add_action('admin_post_xag_save_defaultpostOG', ['MXAG_Seo', 'saveDefaultPostOG']);
			add_action('admin_post_xag_save_taxonomies', ['MXAG_Seo', 'saveTaxonomies']);
			add_action('admin_post_xag_save_miscellaneous', ['MXAG_Seo', 'saveMiscellaneous']);

			// Meta Box
			add_action('add_meta_boxes', ['MXAG_Seo', 'renderBoxes']);

			// Extra Category/Term options
			add_action('category_edit_form_fields', ['MXAG_Seo', 'extraTermFields'], 10, 2);
			add_action('edited_category', ['MXAG_Seo', 'saveExtraTermFields'], 10, 2);

			add_action('post_tag_edit_form_fields', ['MXAG_Seo', 'extraTermFields'], 10, 2);
			add_action('edited_post_tag', ['MXAG_Seo', 'saveExtraTermFields'], 10, 2);

			add_action('init', ['MXAG_Seo', 'getCustomTaxonomies'], 9999);

			// Titles
			add_filter('wp_title', ['MXAG_Seo', 'changeTitle'], 997, 3);
			add_filter('pre_get_document_title', ['MXAG_Seo', 'changeTitle'], 998);
			add_filter('woocommerce_page_title', ['MXAG_Seo', 'changeTitle'], 999, 1);

			// Description
			add_action('wp_head', ['MXAG_Seo', 'changeDescription'], -1000);

			// Open Graph
			add_action('wp_head', ['MXAG_Seo', 'changeOpenGraph']);

			// Meta Robots
			add_action('wp_head', ['MXAG_Seo', 'changeMetaRobots']);

			// Disable WordPress Canonicals
			if (XAG_DISABLE_CANONICALS) {

				// Remove Canonicals
				remove_action('wp_head', 'rel_canonical');

				// remove HTML meta tag
				// <link rel='shortlink' href='http://example.com/?p=25' />
				remove_action('wp_head', 'wp_shortlink_wp_head', 10);

				// remove HTTP header
				// Link: <https://example.com/?p=25>; rel=shortlink
				remove_action('template_redirect', 'wp_shortlink_header', 11);

			}

			// Canonical
			add_action('wp_head', ['MXAG_Seo', 'changeCanonical']);

			// Webmaster Verification
			add_action('wp_head', ['MXAG_Seo', 'webmasterVerification']);

			// Target Keyword
			add_action('wp_head', ['MXAG_Seo', 'forceMetaKeywords'], -1000);

			// Global Scripts
			add_action('wp_head', ['MXAG_Seo', 'renderCustomHeaderScripts']);
			add_action('wp_footer', ['MXAG_Seo', 'renderCustomFooterScripts']);

			// Render Footer
			add_filter('admin_post_xag_footer_links', ['MXAG_Seo', 'renderFreeFooter']);
			add_action('wp_ajax_nopriv_xag_footer_links', ['MXAG_Seo', 'renderFreeFooter']);
			add_action('wp_ajax_xag_footer_links', ['MXAG_Seo', 'renderFreeFooter']);

			// Add Custom Columns for SEO enabled posts
			add_filter('manage_posts_columns', ['MXAG_Seo', 'addCustomColumn']);
			add_action('manage_posts_custom_column', ['MXAG_Seo', 'renderCustomColumn'], 11, 2);
			add_filter('manage_pages_columns', ['MXAG_Seo', 'addCustomColumn']);
			add_action('manage_pages_custom_column', ['MXAG_Seo', 'renderCustomColumn'], 11, 2);

			/// Add new Bulk Actions for SEO enabled posts
			add_action('admin_footer-edit.php', ['MXAG_Seo', 'addBulkActions']);
			add_action('admin_action_xag_seo_enable', ['MXAG_Seo', 'handleBulkAction']);
			add_action('admin_action_xag_seo_disable', ['MXAG_Seo', 'handleBulkAction']);
		}

		public function setDefaultPostSettings()
		{

			// POSTS
			$post_types = get_option('ps_seo_post_types');

			foreach (self::getAllPostObjects() as $post_type) {
				$post_type = (is_array($post_type) ? $post_type['name'] : $post_type);
				if ($post_type !== 'post' && $post_type !== 'page') {
					if (array_key_exists($post_type, $post_types)) {
						continue;
					} else {
						$pa                         = [
							'title'       => '%%title%% %%sep%% %%sitename%%',
							'description' => '',
							'nofollow'    => TRUE
						];
						$post_templates[$post_type] = $pa;
					}
				}
			}

			if (!empty($post_templates) && !empty($post_types)) {
				$post_data = array_merge($post_types, $post_templates);
			} else if (!empty($post_templates)) {
				$post_data = $post_templates;
			}

			if (!empty($post_data)) {
				update_option('ps_seo_post_types', $post_data);
			}
		}

		public function setDefaultTaxonomySettings()
		{

			// TAXONOMIES
			$taxonomy_templates = [];
			$taxonomies         = get_option('ps_seo_taxonomies');

			foreach (self::getAllTaxonomies() as $taxonomy) {
				if (!array_key_exists($taxonomy, $taxonomies)) {
					$pa                            = [
						'title'       => '%%term_title%% %%sep%% %%sitename%%',
						'description' => '',
						'nofollow'    => TRUE
					];
					$taxonomy_templates[$taxonomy] = $pa;
				}
			}

			if (!empty($taxonomy_templates) && !empty($taxonomies)) {
				$taxonomy_data = array_merge($taxonomies, $taxonomy_templates);
			} else if (!empty($taxonomy_templates)) {
				$taxonomy_data = $taxonomy_templates;
			}

			if (!empty($taxonomy_data)) {
				update_option('ps_seo_taxonomies', $taxonomy_data);
			}
		}

		public function extraTermFields($tag, $taxonomy)
		{
			require_once(XAG_PATH . '/pages/metabox/xag_terms.php');
		}

		public static function getCustomTaxonomies()
		{
			$taxonomies = get_taxonomies();
			unset($taxonomies['category']);
			unset($taxonomies['post_tag']);
			unset($taxonomies['nav_menu']);
			unset($taxonomies['link_category']);
			unset($taxonomies['post_format']);

			foreach ($taxonomies as $taxonomy) {
				add_action('edit_' . $taxonomy, ['MXAG_Seo', 'saveExtraTermFields'], 10, 2);
			}

		}

		public function saveExtraTermFields($term_id = 0, $tt_id = 0)
		{

			if (isset($_POST['meta'])) {
				$id       = $term_id;
				$tax      = $_POST['meta']['taxonomy'];
				$cat_meta = get_option($tax . '_' . $id);
				unset($_POST['meta']['taxonomy']);
				$cat_keys = array_keys($_POST['meta']);
				foreach ($cat_keys as $key) {
					if (isset($_POST['meta'][$key])) {
						$cat_meta[$key] = $_POST['meta'][$key];
					}
				}

				/** Schema */
				if (isset($_POST['selectedSchemas'])) {
					$schemaIDs = explode(',', $_POST['selectedSchemas']);
					if (!empty($schemaIDs)) {
						$renderedSchemas = MXAG_Schema::getRemoteRenderedSchemas($schemaIDs, $term_id, 'term');

						if ($renderedSchemas != FALSE) {
							$cat_meta['ps_schema_meta'] = @$renderedSchemas['meta'];
							$cat_meta['ps_schema_data'] = @$renderedSchemas['data'];
						}
					} else {
						$cat_meta['ps_schema_meta'] = FALSE;
						$cat_meta['ps_schema_data'] = FALSE;
					}
				}


				//save the option array
				update_option($tax . '_' . $id, $cat_meta);
			}
		}

		public static function renderCustomHeaderScripts()
		{

			$object = $GLOBALS['wp_query']->get_queried_object();
			if (is_object($object)) {
				$disable_page = get_post_meta($object->ID, 'ps_seo_disable_page_scripts', TRUE);
				
				$disable_global = get_post_meta($object->ID, 'ps_seo_disable_global_scripts', TRUE);

				if (isset($disable_page) && $disable_page != 1) {
					$scripts = get_post_meta($object->ID, 'ps_seo_scripts', TRUE);
					if ($disable_global != 1) {
						$scripts = get_option('ps_seo_global_scripts') . "\n" . $scripts;
					}
				} elseif(isset($disable_global) && $disable_global != 1) {
						$scripts = get_option('ps_seo_global_scripts');
				}

			} else {
				$scripts = get_option('ps_seo_global_scripts');
			}

			if (!empty($scripts)) {
				echo stripslashes_deep($scripts) . "\n";
			}

			// Check if there are shared scripts
			$shared_scripts = get_option('ps_shared_scripts');
			if ($shared_scripts !== FALSE && $shared_scripts !== '') {
				echo stripslashes_deep(base64_decode($shared_scripts)) . "\n";
			}
		}

		public static function renderCustomFooterScripts()
		{

			$object = $GLOBALS['wp_query']->get_queried_object();
			if (is_object($object)) {

				$disable_page = get_post_meta($object->ID, 'ps_seo_disable_page_footer_scripts', TRUE);
				
				$disable_global = get_post_meta($object->ID, 'ps_seo_disable_global_footer_scripts', TRUE);

				if (isset($disable_page) && $disable_page != 1) {
					$scripts = get_post_meta($object->ID, 'ps_seo_footer_scripts', TRUE);
					if ($disable_global != 1) {
						$scripts = get_option('ps_seo_global_footer_scripts') . "\n" . $scripts;
					}
				} elseif(isset($disable_global) && $disable_global != 1) {
						$scripts = get_option('ps_seo_global_footer_scripts');
				}

			} else {
				$scripts = get_option('ps_seo_global_footer_scripts');
			}

			if (!empty($scripts)) {
				echo stripslashes_deep($scripts) . "\n";
			}

			// Check if there are shared scripts
			$shared_scripts = get_option('ps_shared_scripts');
			if ($shared_scripts !== FALSE && $shared_scripts !== '') {
				echo stripslashes_deep(base64_decode($shared_scripts)) . "\n";
			}
		}

		public static function renderFreeFooter()
		{
			$membership = get_option('ps_user_membership');
			$fo         = "";
			if ($membership && !empty($membership)) {
				if ($membership == 'Free Trial' || $membership == 'Forever Free') {
					$http_code = 0;
					if (!$result = get_transient('prs_footer_info')) {
						$result = MXAG_Api::apiRequest(
							$endpoint = 'info',
							$method = 'GET',
							[
								'domain' => preg_replace('/^www\./', '', $_SERVER['SERVER_NAME']),
								'type'   => 'footer_info'
							],
							$http_code
						);
						set_transient('prs_footer_info', $result, 12 * HOUR_IN_SECONDS);
					}

					if (!$result) {
						$fo = "";
					} else {

						if (empty($result['logo'])) {
							$logo = "";
						} else {
							$logo = "<img src='" . $result['logo'] . "' style='width: 32px;margin-top: -11px;' />";
						}

						$fo = '<div id="'. uniqid("psv3_") .'" class="'. uniqid("psv3_") .'" style="display: block !important; clear: both !important">
								<style>
								  * {
								  vertical-align: baseline;
								  font-weight: inherit;
								  font-family: inherit;
								  line-height: 1.3rem;
								  font-style: inherit;
								  font-size: 100%;
								  border: 0 none;
								  outline: 0;
								  padding: 0;
								  margin: 0;
								}

								.PSv3-1 {
								  display: -webkit-flex;
								  display: -ms-flex;
								  display: flex;
								  flex-flow: row nowrap;
								  justify-content: center;
								  align-items: center;
								  align-content: center;
								  font-family: sans-serif;
								  height: 35px;
								  overflow: hidden;
								  width: 100%;
								  clear: both;
								}

								.Psv3-1theme--light {
								  background: #fff;
								  color: #454545;
								}

								.PSv3-1theme--dark {
								  background: #272727;
								  color: #fff;
								}

								.Psv3-1theme--light .PSv3-1link {
								  cursor:pointer;
								  text-decoration: none;
								  color: #454545;
								}

								.PSv3-1theme--dark .PSv3-1link {
								  cursor:pointer;
								  text-decoration: none;
								  color: #fff;
								}

								.PSv3-1logo {
								  text-align: right;
								  padding-right: 20px;
								  align-self: center;
								  line-height:0;
								  border-right: 1px solid rgba(152, 152, 152, 0.45);
								}

								.PSv3-1message {
								  text-align: left;
								  padding-left: 20px;
								  align-self: center;
								}

								.PSv3-1message p {
								  margin: 0;
								}

								@media only screen and (max-width: 767px) {
								  .PSv3-1logo {
								    padding-right: 10px;
								  }
								  .PSv3-1message {
								    padding-left: 10px;
								  }
								}

								@media only screen and (max-width: 549px) {
								  .PSv3-1logo {
								    padding: 0 5px;
								  }
								  .PSv3-1message p {
								    font-size: 12px;
								    padding-right: 5px;
								  }
								}

								@media only screen and (max-width: 320px) and (orientation: portrait) {
								  .PSv3-1message p {
								    font-size: 11px;
								    padding-right: 0px;
								  }
								}

							  </style>

							  <div class="PSv3-1 Psv3-1theme--light" data-nosnippet=""> 
							  <div class="PSv3-1logo"> 
							  <span onclick="window.open(&apos;'. $result['link'] .'&apos;,&apos; _blank &apos;)" class="PSv3-1link"> 
							  <img src="'.$result['logo'].'" alt="Optimized by Xagio. " width="170"> 
							  </span> 
							  </div>
							  <div class="PSv3-1message"> 
							  <p><span onclick="window.open(&apos;'. $result['link'] .'&apos;,&apos; _blank &apos;)" class="PSv3-1link" target="_blank">'. $result['anchor'] .'</span></p> 
							  </div> 
							  </div>

							</div>';
					}
				}
			}
			wp_send_json(['status' => 'error', 'message' => $fo]);
		}

		public static function addBulkActions()
		{

			$post_types = self::getAllPostTypes();

			global $post_type;

			if (in_array($post_type, $post_types)) {
				?>
                <script type="text/javascript">
                    jQuery(document).ready(function () {
                        jQuery("#bulk-action-selector-top").append(
                            '<optgroup label="Xagio">' +
                            '<option value="xag_seo_enable">✓ Enable</option>' +
                            '<option value="xag_seo_disable">✕ Disable</option>' +
                            '</optgroup>'
                        );
                        jQuery("#bulk-action-selector-bottom").append(
                            '<optgroup label="Xagio">' +
                            '<option value="xag_seo_enable">✓ Enable</option>' +
                            '<option value="xag_seo_disable">✕ Disable</option>' +
                            '</optgroup>'
                        );
                    });
                </script>
				<?php
			}
		}

		public static function handleBulkAction()
		{
			$action    = $_REQUEST['action'];
			$post_type = $_REQUEST['post_type'];

			$sendback = admin_url("edit.php?post_type=$post_type");

			$allowed_actions = ["xag_seo_enable", "xag_seo_disable"];
			if (!in_array($action, $allowed_actions)) {
				wp_redirect($sendback);
			}

			$post_ids = $_REQUEST['post'];

			if (empty($post_ids)) {
				wp_redirect($sendback);
			}

			switch ($action) {

				case 'xag_seo_enable':

					foreach ($post_ids as $post_id) {
						update_post_meta($post_id, 'ps_seo_enabled', 1);
					}

					break;
				case 'xag_seo_disable':

					foreach ($post_ids as $post_id) {
						update_post_meta($post_id, 'ps_seo_enabled', 0);
					}

					break;
				default:
					wp_redirect($sendback);
			}

			wp_redirect($sendback);
			exit();
		}

		public static function addCustomColumn($columns)
		{
			return array_merge($columns,
				[
					'project_supremacy_seo' => '<img title="Indicates if Xagio SEO is turned on for this Post/Page." src="' . XAG_URL . 'assets/img/logo-menu-xagio.png"/>',
				]);
		}

		public static function renderCustomColumn($column, $post_id)
		{
			if ($column == 'project_supremacy_seo') {
				$seo_enabled = get_post_meta($post_id, 'ps_seo_enabled', TRUE);
				if ($seo_enabled == 1 || XAG_FORCE_SEO == 1) {
					echo "<i style='color:green'>✓</i>";
				} else {
					echo "<i style='color:red'>✕</i>";
				}
			}
		}

		public static function forceMetaKeywords()
		{

			if (!get_option('ps_seo_target_keyword')) {
				return;
			}

			if (get_option('ps_seo_target_keyword') == "0") {
				return;
			}

			if (!$keyword = MXAG_Seo::getKeywords()) {
				return;
			}

			if (XAG_REMOVE_FOOTPRINT == FALSE) {
				echo "\n<!-- Xagio – Meta Keywords -->\n";
			}

			echo "<meta name=\"keywords\" content=\"$keyword\">";

			if (XAG_REMOVE_FOOTPRINT == FALSE) {
				echo "\n<!-- Xagio – Meta Keywords -->\n\n";
			}
		}

		public static function savePostTypes()
		{
			if (isset($_POST['ps_seo_post_types'])) {
				if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'xag_save_posttypes')) {
					XAG_Init::json('error', 'Sorry, you are not auththorized user.');
				} else {
					$post_types = $_POST['ps_seo_post_types'];
					if (is_array($post_types) && !empty($post_types)) {
						update_option('ps_seo_post_types', $post_types);
					}
					XAG_Init::json('success', 'Your post type settings have been saved.');
				}
			}
			XAG_Init::json('error', 'Sorry, you are not auththorized user.');
		}

		public static function saveCustomPostTypesOG()
		{
			if (isset($_POST['ps_seo_custom_post_types'])) {
				if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'xag_save_customposttypesOG')) {
					XAG_Init::json('error', 'Sorry, you are not auththorized user.');
				} else {
					$post_types = $_POST['ps_seo_custom_post_types'];
					if (is_array($post_types) && !empty($post_types)) {
						update_option('ps_seo_custom_post_types', $post_types);
					}
					XAG_Init::json('success', 'Your post type Open Graph settings have been saved.');
				}
			}
			XAG_Init::json('error', 'Sorry, you are not auththorized user.');
		}

		public static function saveDefaultPostOG()
		{
			if (isset($_POST['ps_seo_default_post_og'])) {
				if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'xag_save_defaultpostOG')) {
					XAG_Init::json('error', 'Sorry, you are not auththorized user.');
				} else {
					$post_types = $_POST['ps_seo_default_post_og'];
					if (is_array($post_types) && !empty($post_types)) {
						update_option('ps_seo_default_post_og', $post_types);
					}
					XAG_Init::json('success', 'Your Default Open Graph settings have been saved.');
				}
			}
			XAG_Init::json('error', 'Sorry, you are not auththorized user.');
		}

		public static function saveTaxonomies()
		{
			if (isset($_POST['ps_seo_taxonomies'])) {
				if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'xag_save_taxonomies')) {
					XAG_Init::json('error', 'Sorry, you are not auththorized user.');
				} else {
					$taxonomies = $_POST['ps_seo_taxonomies'];
					if (is_array($taxonomies) && !empty($taxonomies)) {
						update_option('ps_seo_taxonomies', $taxonomies);
					}
					XAG_Init::json('success', 'Your taxonomy settings have been saved.');
				}
			}
			XAG_Init::json('error', 'Sorry, you are not auththorized user.');
		}

		public static function saveMiscellaneous()
		{
			if (isset($_POST['ps_seo_miscellaneous'])) {
				if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'xag_save_miscellaneous')) {
					XAG_Init::json('error', 'Sorry, you are not auththorized user.');
				} else {
					$miscellaneous = $_POST['ps_seo_miscellaneous'];
					if (is_array($miscellaneous) && !empty($miscellaneous)) {
						update_option('ps_seo_miscellaneous', $miscellaneous);
					}
					XAG_Init::json('success', 'Your miscellaneous settings have been saved.');
				}
			}
			XAG_Init::json('error', 'Sorry, you are not auththorized user.');
		}

		public static function saveGeneral()
		{
			if (!empty($_POST)) {
				if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'xag_save_general')) {
					XAG_Init::json('error', 'Sorry, you are not auththorized user.');
				} else {
					foreach ($_POST as $key => $value) {
						if (strpos($key, 'ps_seo') !== FALSE) {
							if ((isset($_POST['ps_seo_global_scripts']) && !empty($_POST['ps_seo_global_scripts'])) OR (isset($_POST['ps_seo_global_footer_scripts']) && !empty($_POST['ps_seo_global_footer_scripts'])) ) {
								update_option($key, trim($value));
							} else {
								update_option($key, xag_stripAllSlashes(htmlspecialchars(trim($value))));
							}
						}
					}
					XAG_Init::json('success', 'Your general settings have been saved.');
				}
			}
			XAG_Init::json('error', 'Sorry, you are not auththorized user.');
		}

		public static function webmasterVerification()
		{
			$webmaster_template = [];

			$bing             = get_option('ps_seo_verify_bing');
			$google           = get_option('ps_seo_verify_google');
			$google_analytics = get_option('ps_seo_verify_google_analytics');
			$pinterest        = get_option('ps_seo_verify_pinterest');
			$yandex           = get_option('ps_seo_verify_yandex');

			if (!empty($bing)) {
				$webmaster_template[] = '<meta name="msvalidate.01" content="' . esc_attr($bing) . '" />';
			}
			if (!empty($google)) {
				$webmaster_template[] = '<meta name="google-site-verification" content="' . esc_attr($google) . '"/>';
			}
			if (!empty($google_analytics)) {

				$webmaster_template[] = '<script async src="https://www.googletagmanager.com/gtag/js?id=' . esc_attr($google_analytics) . '"></script>';
				$webmaster_template[] = '<script>';
				$webmaster_template[] = '  window.dataLayer = window.dataLayer || [];';
				$webmaster_template[] = '  function gtag(){dataLayer.push(arguments);}';
				$webmaster_template[] = '  gtag(\'js\', new Date());';
				$webmaster_template[] = '  gtag(\'config\', \'' . esc_attr($google_analytics) . '\');';
				$webmaster_template[] = '</script>';


			}
			if (!empty($pinterest)) {
				$webmaster_template[] = '<meta name="p:domain_verify" content="' . esc_attr($pinterest) . '"/>';
			}
			if (!empty($yandex)) {
				$webmaster_template[] = '<meta name="yandex-verification" content="' . esc_attr($yandex) . '" />';
			}
			if (sizeof($webmaster_template) > 0) {
				if (XAG_REMOVE_FOOTPRINT == FALSE) {
					echo "\n<!-- Xagio – Webmaster Verification -->\n";
				}

				echo join("\n", $webmaster_template);

				if (XAG_REMOVE_FOOTPRINT == FALSE) {
					echo "\n<!-- Xagio – Webmaster Verification -->\n\n";
				}
			}
		}

		public static function changeCanonical()
		{

			$canonical = '';
			if (XAG_DISABLE_CANONICALS) {
				$canonical = get_site_url() . $_SERVER['REQUEST_URI'];
			}

			/**
			 *  Try to get a custom canonical if set
			 */
			if (is_singular()) {

				$page_canonical = MXAG_Seo::getCanonical();
				if ($page_canonical != FALSE) {
					$canonical = $page_canonical;
				}

			}

			if (!empty($canonical)) {
				if (XAG_REMOVE_FOOTPRINT == FALSE) {
					echo "\n<!-- Xagio – Canonical URL -->\n";
				}

				echo '<link rel="canonical" href="' . esc_attr($canonical) . '" />';

				if (XAG_REMOVE_FOOTPRINT == FALSE) {
					echo "\n<!-- Xagio – Canonical URL -->\n";
				}
			}

			return TRUE;

		}

		public static function changeMetaRobots()
		{
			if (is_feed()) {
				return FALSE;
			}

			$robots = '';

			if (is_singular()) {

				$robots = MXAG_Seo::getRobots();

			} else if (is_category() || is_tag() || is_tax()) {

				$robots = MXAG_Seo::getRobotsTaxonomy();

			} else if (is_search()) {

				$robots = MXAG_Seo::getRobotsMisc();

			} else if (is_author()) {

				$robots = MXAG_Seo::getRobotsMisc();

			} else if (is_post_type_archive()) {

				$robots = MXAG_Seo::getRobotsMisc();

			} else if (is_archive()) {

				$robots = MXAG_Seo::getRobotsMisc();

			} else if (is_404()) {

				$robots = MXAG_Seo::getRobotsMisc();

			}


			if (get_option('ps_seo_force_noodp') == "1") {
				if (empty($robots)) {
					$robots = 'noodp';
				} else {
					$robots   = explode(',', $robots);
					$robots[] = 'noodp';
					$robots   = join(',', $robots);
				}
			}

			if (get_option('ps_seo_index_subpages') == "1") {
				if (MXAG_Seo::is_sub_page()) {
					if (empty($robots)) {
						$robots = 'noindex';
					} else {
						$robots   = explode(',', $robots);
						$robots   = array_diff($robots, ["index", "noindex"]);
						$robots[] = 'noindex';
						$robots   = join(',', $robots);
					}
				}
			}

			if (!empty($robots)) {
				if (XAG_REMOVE_FOOTPRINT == FALSE) {
					echo "\n<!-- Xagio – Meta Robots -->\n";
				}

				echo "<meta name='robots' content='$robots'/>";

				if (XAG_REMOVE_FOOTPRINT == FALSE) {
					echo "\n<!-- Xagio – Meta Robots -->\n";
				}
			}

			return TRUE;
		}

		public static function getRobots()
		{
			$object = $GLOBALS['wp_query']->get_queried_object();
			if (is_object($object)) {

				// Load all variables
				$meta = MXAG_Seo::formatMetaVariables(get_post_meta($object->ID));

				if (@$meta['ps_seo_enabled'] == 1 || XAG_FORCE_SEO == 1) {

					if (@$meta['ps_seo_metarobots_enabled'] == TRUE) {
						$robots = explode(',', @$meta['ps_seo_metarobots_advanced']);
						if (!is_array($robots) || sizeof($robots) < 1) {
							$robots = [];
						}
						$robots[] = @$meta['ps_seo_metarobots_index'];
						$robots[] = @$meta['ps_seo_metarobots_follow'];
						return join(',', $robots);
					}

				} else {

					$post_type  = (isset($object->post_type) ? $object->post_type : $object->query_var);
					$post_types = get_option('ps_seo_post_types');

					$robots = @$post_types[$post_type]['nofollow'];
					if ($robots == TRUE) {
						return 'noindex,follow';
					} else {
						return FALSE;
					}
				}
			} else {
				return FALSE;
			}
		}

		public static function getRobotsTaxonomy()
		{
			$object = $GLOBALS['wp_query']->get_queried_object();
			if (is_object($object)) {

				$taxonomy   = $object->taxonomy;
				$taxonomies = get_option('ps_seo_taxonomies');

				$robots = @$taxonomies[$taxonomy]['nofollow'];

				$meta = get_option($taxonomy . '_' . $object->term_id);
				if (@$meta['term_seo_nofollow'] == TRUE) {

					return 'noindex,follow';

				} else if ($robots == TRUE) {

					return 'noindex,follow';

				} else {

					return FALSE;

				}

			} else {
				return FALSE;
			}
		}

		private static function getCurrentMisc()
		{
			if (is_search()) {
				return 'search';
			} else if (is_author()) {
				return 'author';
			} else if (is_post_type_archive()) {
				return 'archive_post';
			} else if (is_archive()) {
				return 'archive';
			} else if (is_404()) {
				return 'not_found';
			} else {
				return FALSE;
			}
		}

		public static function getRobotsMisc()
		{
			$miscellaneous = get_option('ps_seo_miscellaneous');
			$misc          = self::getCurrentMisc();
			if (isset($miscellaneous[$misc])) {
				$robots = @$miscellaneous[$misc]['nofollow'];
				if ($robots == TRUE) {
					return 'noindex,follow';
				} else {
					return FALSE;
				}
			} else {
				return FALSE;
			}
		}

		public static function changeOpenGraph()
		{
			if (is_feed()) {
				return FALSE;
			}

			$og = '';

			if (MXAG_Seo::is_home_static_page()) {

				$og = MXAG_Seo::getOGHome();

			} else if (MXAG_Seo::is_home_posts_page()) {

				$og = MXAG_Seo::getOGHome();

			} else if (is_singular()) {

				$og = MXAG_Seo::getOG();

			}

			if (!empty($og)) {
				if (XAG_REMOVE_FOOTPRINT == FALSE) {
					echo "\n<!-- Xagio – Open Graph -->\n";
				}

				echo $og;

				if (XAG_REMOVE_FOOTPRINT == FALSE) {
					echo "\n<!-- Xagio – Open Graph -->\n";
				}
			}

			return TRUE;
		}

		public static function getOG()
		{
			$object = $GLOBALS['wp_query']->get_queried_object();
			if (is_object($object)) {
				// Load all variables
				$meta = MXAG_Seo::formatMetaVariables(get_post_meta($object->ID));

				if (isset($meta['ps_seo_enabled']) && $meta['ps_seo_enabled'] != TRUE) {
					return FALSE;
				}

				$og    = '';
				$fb_OG = [
					'ID'          => self::replaceVars(@$meta['ps_seo_fb_app_id']),
					'Title'       => self::replaceVars(@$meta['ps_seo_fb_title']),
					'Description' => self::replaceVars(@$meta['ps_seo_fb_desc']),
					'Image'       => self::replaceVars(@$meta['ps_seo_fb_img'])
				];
				$tw_OG = [
					'Title'       => self::replaceVars(@$meta['ps_seo_tw_title']),
					'Description' => self::replaceVars(@$meta['ps_seo_tw_desc']),
					'Image'       => self::replaceVars(@$meta['ps_seo_tw_img'])
				];

				$post_type         = (isset($object->post_type) ? $object->post_type : $object->query_var);
				$custom_post_types = get_option('ps_seo_custom_post_types');
				$default_og        = get_option('ps_seo_default_post_og');
				$default_og_status = get_option('prs_default_og_status');

				if (!empty($fb_OG['Title']) && !empty($fb_OG['Description']) && !empty($fb_OG['Image'])) {
					$og .= '<meta property="og:locale" content="en_US"/>' . "\n";
					$og .= '<meta property="og:type" content="article"/>' . "\n";
					$og .= '<meta property="fb:app_id" content="' . @$fb_OG['ID'] . '"/>' . "\n";
					$og .= '<meta property="og:title" content="' . stripslashes($fb_OG['Title']) . '"/>' . "\n";
					$og .= '<meta property="og:description" content="' . stripslashes($fb_OG['Description']) . '"/>' . "\n";
					$og .= '<meta property="og:url" content="' . self::getScheme() . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] . '"/>' . "\n";
					$og .= '<meta property="og:site_name" content="' . get_bloginfo('name') . '"/>' . "\n";
					$og .= '<meta property="og:image" content="' . $fb_OG['Image'] . '" />';
				} else {

					if (isset($custom_post_types)) {
						$facebook_custom_title       = @$custom_post_types[$post_type]['facebook_title'];
						$facebook_custom_description = @$custom_post_types[$post_type]['facebook_description'];
						$facebook_custom_image       = @$custom_post_types[$post_type]['facebook_image'];
					}
					$facebook_title       = '';
					$facebook_description = '';
					if (!empty($facebook_custom_title) && !empty($facebook_custom_description) && !empty($facebook_custom_image)) {
						$facebook_title       = $facebook_custom_title;
						$facebook_description = $facebook_custom_description;
						$facebook_image       = $facebook_custom_image;
					}

					if ($default_og_status != 1) {
						if (isset($default_og)) {
							$facebook_default_title       = @$default_og[$post_type]['facebook_title'];
							$facebook_default_app_id      = @$default_og[$post_type]['facebook_app_id'];
							$facebook_default_description = @$default_og[$post_type]['facebook_description'];
							$facebook_default_image       = @$default_og[$post_type]['facebook_image'];
						}

						if (!empty($facebook_default_title) && !empty($facebook_default_description) && !empty($facebook_default_image)) {
							$facebook_app_id      = $facebook_default_app_id;
							$facebook_title       = $facebook_default_title;
							$facebook_description = $facebook_default_description;
							$facebook_image       = $facebook_default_image;
						}
					}

					///////////////////////////////////////// Variables $facebook_title AND $facebook_description HAS TO BE DEFINED BEFORE ANY IF STATEMENTS
					$facebook_title       = self::replaceVars($facebook_title, $object->ID);
					$facebook_description = self::replaceVars($facebook_description, $object->ID);

					if (!empty($facebook_title) && !empty($facebook_description) && !empty($facebook_image)) {
						$og .= '<meta property="og:locale" content="en_US"/>' . "\n";
						$og .= '<meta property="og:type" content="article"/>' . "\n";
						$og .= '<meta property="fb:app_id" content="' . @$facebook_app_id . '"/>' . "\n";
						$og .= '<meta property="og:title" content="' . stripslashes($facebook_title) . '"/>' . "\n";
						$og .= '<meta property="og:description" content="' . stripslashes($facebook_description) . '"/>' . "\n";
						$og .= '<meta property="og:url" content="' . self::getScheme() . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] . '"/>' . "\n";
						$og .= '<meta property="og:site_name" content="' . get_bloginfo('name') . '"/>' . "\n";
						$og .= '<meta property="og:image" content="' . $facebook_image . '" />';
					}
				}

				if (!empty($tw_OG['Title']) && !empty($tw_OG['Description']) && !empty($tw_OG['Image'])) {
					$og .= '<meta name="twitter:card" content="summary"/>' . "\n";
					$og .= '<meta name="twitter:title" content="' . stripslashes($tw_OG['Title']) . '"/>' . "\n";
					$og .= '<meta name="twitter:description" content="' . stripslashes($tw_OG['Description']) . '"/>' . "\n";
					$og .= '<meta name="twitter:image" content="' . $tw_OG['Image'] . '"/>';
				} else {

					if (isset($custom_post_types)) {
						$twitter_custom_title       = @$custom_post_types[$post_type]['twitter_title'];
						$twitter_custom_description = @$custom_post_types[$post_type]['twitter_description'];
						$twitter_custom_image       = @$custom_post_types[$post_type]['twitter_image'];
					}

					$twitter_title       = '';
					$twitter_description = '';
					if (!empty($twitter_custom_title) && !empty($twitter_custom_description) && !empty($twitter_custom_image)) {
						$twitter_title       = $twitter_custom_title;
						$twitter_description = $twitter_custom_description;
						$twitter_image       = $twitter_custom_image;
					}

					if ($default_og_status != 1) {
						if (isset($default_og)) {
							$twitter_default_title       = @$default_og[$post_type]['twitter_title'];
							$twitter_default_description = @$default_og[$post_type]['twitter_description'];
							$twitter_default_image       = @$default_og[$post_type]['twitter_image'];
						}

						if (!empty($twitter_default_title) && !empty($twitter_default_description) && !empty($twitter_default_image)) {
							$twitter_title       = $twitter_default_title;
							$twitter_description = $twitter_default_description;
							$twitter_image       = $twitter_default_image;
						}
					}
					///////////////////////////////////////// Variables $twitter_title AND $twitter_description HAS TO BE DEFINED BEFORE ANY IF STATEMENTS
					$twitter_title       = self::replaceVars($twitter_title, $object->ID);
					$twitter_description = self::replaceVars($twitter_description, $object->ID);

					if (!empty($twitter_title) && !empty($twitter_description) && !empty($twitter_image)) {
						$og .= '<meta name="twitter:card" content="summary"/>' . "\n";
						$og .= '<meta name="twitter:title" content="' . stripslashes($twitter_title) . '"/>' . "\n";
						$og .= '<meta name="twitter:description" content="' . stripslashes($twitter_description) . '"/>' . "\n";
						$og .= '<meta name="twitter:image" content="' . $twitter_image . '"/>';
					}

				}

				return $og;
			}

			return FALSE;
		}

		public static function getOGHome()
		{

			$og = self::getOG();
			if (!empty($og)) {
				return $og;
			}

			$og    = '';
			$fb_OG = [
				'ID'          => self::replaceVars(get_option('ps_seo_app_id_fb')),
				'Title'       => self::replaceVars(get_option('ps_seo_title_fb')),
				'Description' => self::replaceVars(get_option('ps_seo_description_fb')),
				'Image'       => self::replaceVars(get_option('ps_seo_image_fb'))
			];
			$tw_OG = [
				'Title'       => self::replaceVars(get_option('ps_seo_title_tw')),
				'Description' => self::replaceVars(get_option('ps_seo_description_tw')),
				'Image'       => self::replaceVars(get_option('ps_seo_image_tw'))
			];
			if (!empty($fb_OG['Title']) && !empty($fb_OG['Description']) && !empty($fb_OG['Image'])) {
				$og .= '<meta property="og:locale" content="en_US"/>' . "\n";
				$og .= '<meta property="og:type" content="article"/>' . "\n";
				$og .= '<meta property="fb:app_id" content="' . @$fb_OG['ID'] . '"/>' . "\n";
				$og .= '<meta property="og:title" content="' . stripslashes($fb_OG['Title']) . '"/>' . "\n";
				$og .= '<meta property="og:description" content="' . stripslashes($fb_OG['Description']) . '"/>' . "\n";
				$og .= '<meta property="og:url" content="' . self::getScheme() . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] . '"/>' . "\n";
				$og .= '<meta property="og:site_name" content="' . get_bloginfo('name') . '"/>' . "\n";
				$og .= '<meta property="og:image" content="' . $fb_OG['Image'] . '" />';
			}
			if (!empty($tw_OG['Title']) && !empty($tw_OG['Description']) && !empty($tw_OG['Image'])) {
				$og .= '<meta name="twitter:card" content="summary"/>' . "\n";
				$og .= '<meta name="twitter:title" content="' . stripslashes($tw_OG['Title']) . '"/>' . "\n";
				$og .= '<meta name="twitter:description" content="' . stripslashes($tw_OG['Description']) . '"/>' . "\n";
				$og .= '<meta name="twitter:image" content="' . $tw_OG['Image'] . '"/>';
			}
			return $og;
		}

		public static function getScheme()
		{
			if ((!empty($_SERVER['REQUEST_SCHEME']) && $_SERVER['REQUEST_SCHEME'] == 'https') || (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] == 'on') || (!empty($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == '443')) {
				return 'https';
			} else {
				return 'http';
			}
		}

		public static function changeDescription()
		{

			if (is_feed()) {
				return FALSE;
			}

			$description = '';

			if (MXAG_Seo::is_home_static_page()) {

				$description = MXAG_Seo::getDescriptionHome();

			} else if (MXAG_Seo::is_home_posts_page()) {

				$description = MXAG_Seo::getDescriptionHome();

			} else if (is_singular()) {

				$description = MXAG_Seo::getDescription();

			} else if (is_search()) {

				$description = MXAG_Seo::getMiscDescription();

			} else if (is_category() || is_tag() || is_tax()) {

				$description = MXAG_Seo::getTermDescription();

			} else if (is_author()) {

				$description = MXAG_Seo::getMiscDescription();

			} else if (is_post_type_archive()) {

				$description = MXAG_Seo::getMiscDescription();

			} else if (is_archive()) {

				$description = MXAG_Seo::getMiscDescription();

			} else if (is_404()) {

				$description = MXAG_Seo::getMiscDescription();

			}

			if (!empty($description)) {
				if (XAG_REMOVE_FOOTPRINT == FALSE) {
					echo "\n<!-- Xagio – Meta Description -->\n";
				}

				echo '<meta name="description" content="' . esc_attr($description) . '">';

				if (XAG_REMOVE_FOOTPRINT == FALSE) {
					echo "\n<!-- Xagio – Meta Description -->\n";
				}
			}

			return TRUE;
		}

		public static function getKeywords()
		{
			$object = $GLOBALS['wp_query']->get_queried_object();
			if (is_object($object)) {
				// Load all variables
				$meta = MXAG_Seo::formatMetaVariables(get_post_meta($object->ID));

				if (@$meta['ps_seo_enabled'] != 1 && XAG_FORCE_SEO != 1) {
					return FALSE;
				}

				if (!empty($meta['ps_seo_keyword'])) {
					return self::replaceVars($meta['ps_seo_keyword'], $object->ID);
				}

				return FALSE;
			}

			return FALSE;
		}

		public static function getCanonical()
		{
			$object = $GLOBALS['wp_query']->get_queried_object();
			if (is_object($object)) {
				// Load all variables
				$meta = MXAG_Seo::formatMetaVariables(get_post_meta($object->ID));

				if (@$meta['ps_seo_enabled'] != 1 && XAG_FORCE_SEO != 1) {
					return FALSE;
				}

				if (!empty($meta['ps_seo_canonical'])) {
					return $meta['ps_seo_canonical'];
				}

				return FALSE;
			}

			return FALSE;
		}

		public static function getDescription()
		{
			$object = $GLOBALS['wp_query']->get_queried_object();
			if (is_object($object)) {
				// Load all variables
				$meta = MXAG_Seo::formatMetaVariables(get_post_meta($object->ID));

				if (@$meta['ps_seo_enabled'] != 1 && XAG_FORCE_SEO != 1) {
					return FALSE;
				}

				if (!empty($meta['ps_seo_description'])) {
					return self::replaceVars($meta['ps_seo_description'], $object->ID);
				} else {
					$post_type  = (isset($object->post_type) ? $object->post_type : $object->query_var);
					$post_types = get_option('ps_seo_post_types');

					$template = @$post_types[$post_type]['description'];
					$template = self::replaceVars($template, $object->ID);

					if (!empty($template)) {
						return $template;
					}
				}

				return FALSE;
			}

			return FALSE;
		}

		public static function getTermDescription()
		{
			$object = $GLOBALS['wp_query']->get_queried_object();
			if (is_object($object)) {

				$taxonomies = get_option('ps_seo_taxonomies');
				$taxonomy   = $object->taxonomy;

				$meta = get_option($taxonomy . '_' . $object->term_id);

				if (isset($meta['term_seo_description']) && !empty($meta['term_seo_description'])) {
					return MXAG_Seo::replaceVars($meta['term_seo_description']);
				}
				if (isset($taxonomies[$taxonomy])) {
					return MXAG_Seo::replaceVars($taxonomies[$taxonomy]['description']);
				} else {
					return FALSE;
				}

			} else {
				return FALSE;
			}
		}

		public static function getMiscDescription()
		{
			$miscellaneous = get_option('ps_seo_miscellaneous');
			$misc          = self::getCurrentMisc();
			if (isset($miscellaneous[$misc])) {
				return MXAG_Seo::replaceVars($miscellaneous[$misc]['description']);
			} else {
				return FALSE;
			}
		}

		public static function getDescriptionHome()
		{

			$description = self::getDescription();
			if (empty($description)) {
				$description = get_option('ps_seo_description');
			}

			return self::replaceVars($description);
		}

		public static function changeTitle($title, $separator = '', $separator_location = '')
		{
			// Ignore Feeds
			if (is_feed()) {
				return $title;
			}

			// Original Title
			$original_title = $title;

			if (MXAG_Seo::is_home_static_page()) {

				$title = MXAG_Seo::getTitleHome();
				if (!is_string($title) || '' === $title) {
					$title = $original_title;
				}

			} else if (MXAG_Seo::is_home_posts_page()) {

				$title = MXAG_Seo::getTitleHome();
				if (empty($title)) {
					$title = $original_title;
				}

			} else if (is_singular()) {

				$title = MXAG_Seo::getTitle();

				if (!is_string($title) || '' === $title) {
					$title = $original_title;
				}

			} else if (is_search()) {

				$title = MXAG_Seo::getMiscTitle();

				if (!is_string($title) || '' === $title) {
					$title = $original_title;
				}

			} else if (is_category() || is_tag() || is_tax()) {

				$title = MXAG_Seo::getTermTitle();

				if (!is_string($title) || '' === $title) {
					$title = $original_title;
				}

			} else if (is_author()) {

				$title = MXAG_Seo::getMiscTitle();

				if (!is_string($title) || '' === $title) {
					$title = $original_title;
				}

			} else if (is_post_type_archive()) {

				$title = MXAG_Seo::getMiscTitle();

				if (!is_string($title) || '' === $title) {
					$title = $original_title;
				}

			} else if (is_archive()) {

				$title = MXAG_Seo::getMiscTitle();

				if (!is_string($title) || '' === $title) {
					$title = $original_title;
				}

			} else if (is_404()) {

				$title = MXAG_Seo::getMiscTitle();

				if (!is_string($title) || '' === $title) {
					$title = $original_title;
				}

			} else {

				$title = MXAG_Seo::getMiscTitle();

				if (!is_string($title) || '' === $title) {
					$title = $original_title;
				}

			}

			return $title;
		}

		public static function replaceVars($string, $post_id = 0, $pre_replace = [])
		{
            if (empty($string)) return '';

			global $wp;

			if (is_array($string)) {
				if (sizeof($string) > 0) {
					$string = $string[0];
				} else {
					$string = '';
				}
			}

			include_once(ABSPATH . 'wp-admin/includes/plugin.php');
			$theme = wp_get_theme();

			if (is_plugin_active('wpglow-builder/wpglow-builder.php') || is_plugin_active('tf-numbers-number-counter-animaton/tf-random_numbers.php') || 'Pro Styler Rev' == $theme->name || 'PS Theme Maker' == $theme->name) {
				$vars = [
					'%%sitename%%'           => get_bloginfo('name'),
					'%%tagline%%'            => get_bloginfo('description'),
					'%%siteurl%%'            => get_site_url(),
					'%%currurl%%'            => home_url(add_query_arg([], $wp->request)),
					'%%sep%%'                => (!get_option('ps_seo_title_separator')) ? '-' : get_option('ps_seo_title_separator'),
					'%%title%%'              => ($post_id !== 0) ? get_the_title($post_id) : '',
					'%%parent_title%%'       => ($post_id !== 0) ? get_the_title(wp_get_post_parent_id($post_id)) : '',
					'%%date%%'               => get_the_date('', $post_id),
					'%%pretty_date%%'        => get_the_date('F Y', $post_id),
					'%%tag%%'                => ($post_id !== 0) ? self::getPostTags($post_id) : '',
					'%%category%%'           => ($post_id !== 0) ? self::getPostCategories($post_id) : '',
					'%%category_primary%%'   => ($post_id !== 0) ? self::getPostCategoryPrimary($post_id) : '',
					'%%term_title%%'         => (is_category() || is_tag() || is_tax()) ? @$GLOBALS['wp_query']->get_queried_object()->name : '',
					'%%search_query%%'       => get_search_query(TRUE),
					'%%ps_seo_title%%'       => ($post_id !== 0) ? get_post_meta($post_id, 'ps_seo_title', TRUE) : '',
					'%%ps_seo_description%%' => ($post_id !== 0) ? get_post_meta($post_id, 'ps_seo_description', TRUE) : '',
					'%%author_name%%'        => get_the_author(),
				];
			} else {
				$vars = [
					'%%sitename%%'           => get_bloginfo('name'),
					'%%tagline%%'            => get_bloginfo('description'),
					'%%siteurl%%'            => get_site_url(),
					'%%currurl%%'            => home_url(add_query_arg([], $wp->request)),
					'%%sep%%'                => (!get_option('ps_seo_title_separator')) ? '-' : get_option('ps_seo_title_separator'),
					'%%title%%'              => ($post_id !== 0) ? get_the_title($post_id) : '',
					'%%parent_title%%'       => ($post_id !== 0) ? get_the_title(wp_get_post_parent_id($post_id)) : '',
					'%%date%%'               => get_the_date('', $post_id),
					'%%pretty_date%%'        => get_the_date('F Y', $post_id),
					'%%excerpt%%'            => ($post_id !== 0) ? get_the_excerpt($post_id) : '',
					'%%tag%%'                => ($post_id !== 0) ? self::getPostTags($post_id) : '',
					'%%category%%'           => ($post_id !== 0) ? self::getPostCategories($post_id) : '',
					'%%category_primary%%'   => ($post_id !== 0) ? self::getPostCategoryPrimary($post_id) : '',
					'%%term_title%%'         => (is_category() || is_tag() || is_tax()) ? @$GLOBALS['wp_query']->get_queried_object()->name : '',
					'%%search_query%%'       => get_search_query(TRUE),
					'%%ps_seo_title%%'       => ($post_id !== 0) ? get_post_meta($post_id, 'ps_seo_title', TRUE) : '',
					'%%ps_seo_description%%' => ($post_id !== 0) ? get_post_meta($post_id, 'ps_seo_description', TRUE) : '',
					'%%author_name%%'        => get_the_author(),
				];
			}
			foreach ($pre_replace as $name => $value) {
				if (is_array($name)) {
					$name = $name[0];
				}
				if (is_array($value)) {
					$value = $value[0];
				}
				$string = str_replace($name, $value, $string);
			}
			foreach ($vars as $name => $value) {
				if (is_array($name)) {
					$name = $name[0];
				}
				if (is_array($value)) {
					$value = $value[0];
				}
				$string = str_replace($name, $value, $string);
			}
			return $string;
		}

		public static function getMiscTitle()
		{
			$miscellaneous = get_option('ps_seo_miscellaneous');
			$misc          = self::getCurrentMisc();
			if (isset($miscellaneous[$misc])) {
				return MXAG_Seo::replaceVars($miscellaneous[$misc]['title']);
			} else {
				return FALSE;
			}
		}

		public static function getTermTitle()
		{
			$object = $GLOBALS['wp_query']->get_queried_object();
			if (is_object($object)) {

				$taxonomies = get_option('ps_seo_taxonomies');
				$taxonomy   = $object->taxonomy;

				$meta = get_option($taxonomy . '_' . $object->term_id);

				if (isset($meta['term_seo_title']) && !empty($meta['term_seo_title'])) {
					return MXAG_Seo::replaceVars($meta['term_seo_title']);
				} else if (isset($taxonomies[$taxonomy])) {
					return MXAG_Seo::replaceVars($taxonomies[$taxonomy]['title']);
				} else {
					return FALSE;
				}

			} else {
				return FALSE;
			}
		}

		public static function getTitleHome()
		{

			$title = self::getTitle();

			if (empty($title)) {
				$title = stripslashes_deep(get_option('ps_seo_title'));
			}

			return self::replaceVars($title);
		}

		public static function getTitle()
		{

			$object = $GLOBALS['wp_query']->get_queried_object();

			if (is_object($object)) {

				// Load all variables
				$meta = MXAG_Seo::formatMetaVariables(get_post_meta($object->ID));

				if (@$meta['ps_seo_enabled'] != 1 && XAG_FORCE_SEO != 1) {
					return FALSE;
				}

				if (!empty($meta['ps_seo_title'])) {
					return self::replaceVars($meta['ps_seo_title'], $object->ID);
				} else {
					$post_type  = (isset($object->post_type) ? $object->post_type : $object->query_var);
					$post_types = get_option('ps_seo_post_types');

					$template = @$post_types[$post_type]['title'];
					$template = self::replaceVars($template, $object->ID);

					if (!empty($template)) {
						return $template;
					}
				}

				return FALSE;
			}

			return FALSE;
		}

		public static function renderBoxes($post_type)
		{
			if (!get_option('ps_hidden')) {
				if (in_array($post_type, self::getAllPostTypes())) {
					add_meta_box(
						'xag_seo'
						, 'Xagio - On Page SEO'
						, ['MXAG_Seo', 'renderSEO']
						, $post_type
						, 'advanced'
						, 'high'
					);
				}
			}
		}

		public static function save($post_id)
		{

			// Fix for trashing posts/pages
			if (!isset($_POST['post_ID'])) {
				return $post_id;
			}
			// Fix for Fusion Builder page ID
			if ($_POST['post_ID'] != $post_id) {
				$post_id = $_POST['post_ID'];
			}

			// Check if our nonce is set.
			if (!isset($_POST['xag_nonce'])) {
				return $post_id;
			}

			if (wp_is_post_revision($post_id)) {
				return $post_id;
			}

			if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
				return $post_id;
			}

			// Check the user's permissions.
			if ('page' === $_POST['post_type']) {

				if (!current_user_can('edit_page', $post_id)) {
					return $post_id;
				}

			} else {

				if (!current_user_can('edit_post', $post_id)) {
					return $post_id;
				}
			}

			/**
			 *  BEGIN THE SAVING PROCESS
			 */

			// Shortcode support
			$_POST['ps_seo_title']       = trim(strip_tags(do_shortcode($_POST['ps_seo_title'])));
			$_POST['ps_seo_description'] = trim(strip_tags(do_shortcode($_POST['ps_seo_description'])));

			// Change the URL if modified
			$newUrl = self::extract_url($post_id);
			$oriUrl = $_POST['ps_seo_url'];

			// ONLY PERFORM ON UPDATES
			if (
				$_POST['post_status'] === 'publish' &&
				$_POST['original_post_status'] === 'publish' &&
				@$_POST['save'] === 'Update' &&
				get_option('ps_disable_redirects') != TRUE
			) {
				if ($newUrl != $oriUrl) {
					if (XAG_DISABLE_REDIRECTS != TRUE) {
						MXAG_Redirects::add($oriUrl, $newUrl);
					}
				}
			}

			// Update the URL in Group
			MXAG_Groups::updateData(
				[
					'url'         => $newUrl,
					'title'       => @$_POST['ps_seo_title'],
					'description' => @$_POST['ps_seo_description'],
					'notes'       => @$_POST['ps_seo_notes'],
					'h1'          => @$_POST['post_title']
				],
				[
					'id_page_post' => $post_id
				]);

			/** SEO */
			update_post_meta($post_id, 'ps_seo_enabled', @$_POST['ps_seo_enabled']);
			update_post_meta($post_id, 'ps_seo_title', @$_POST['ps_seo_title']);
			update_post_meta($post_id, 'ps_seo_description', @$_POST['ps_seo_description']);
			update_post_meta($post_id, 'ps_seo_url', $newUrl);
			update_post_meta($post_id, 'ps_seo_keyword', @$_POST['ps_seo_keyword']);
			update_post_meta($post_id, 'ps_seo_notes', @$_POST['ps_seo_notes']);
			update_post_meta($post_id, 'ps_seo_scripts', @wp_slash($_POST['ps_seo_scripts']));
			update_post_meta($post_id, 'ps_seo_footer_scripts', @wp_slash($_POST['ps_seo_footer_scripts']));
			update_post_meta($post_id, 'ps_seo_disable_global_scripts', @$_POST['ps_seo_disable_global_scripts']);
			update_post_meta($post_id, 'ps_seo_disable_global_footer_scripts', @$_POST['ps_seo_disable_global_footer_scripts']);
			update_post_meta($post_id, 'ps_seo_disable_page_scripts', @$_POST['ps_seo_disable_page_scripts']);
			update_post_meta($post_id, 'ps_seo_disable_page_footer_scripts', @$_POST['ps_seo_disable_page_footer_scripts']);


			update_post_meta($post_id, 'ps_seo_metarobots_enabled', @$_POST['ps_seo_metarobots_enabled']);
			update_post_meta($post_id, 'ps_seo_metarobots_index', @$_POST['ps_seo_metarobots_index']);
			update_post_meta($post_id, 'ps_seo_metarobots_follow', @$_POST['ps_seo_metarobots_follow']);

			// ps_seo_metarobots_advanced is an array?

			if (!is_null(@$_POST['ps_seo_metarobots_advanced'])) {
				$_POST['ps_seo_metarobots_advanced'] = join(',', $_POST['ps_seo_metarobots_advanced']);
			}

			update_post_meta($post_id, 'ps_seo_metarobots_advanced', @$_POST['ps_seo_metarobots_advanced']);

			update_post_meta($post_id, 'ps_seo_canonical', @$_POST['ps_seo_canonical']);

			update_post_meta($post_id, 'ps_seo_tw_title', @$_POST['ps_seo_tw_title']);
			update_post_meta($post_id, 'ps_seo_tw_desc', @$_POST['ps_seo_tw_desc']);
			update_post_meta($post_id, 'ps_seo_tw_img', @$_POST['ps_seo_tw_img']);

			update_post_meta($post_id, 'ps_seo_fb_app_id', @$_POST['ps_seo_fb_app_id']);
			update_post_meta($post_id, 'ps_seo_fb_title', @$_POST['ps_seo_fb_title']);
			update_post_meta($post_id, 'ps_seo_fb_desc', @$_POST['ps_seo_fb_desc']);
			update_post_meta($post_id, 'ps_seo_fb_img', @$_POST['ps_seo_fb_img']);


			/** Schema */
			if (isset($_POST['selectedSchemas'])) {
				$schemaIDs = explode(',', $_POST['selectedSchemas']);
				if (!empty($schemaIDs)) {
					$renderedSchemas = MXAG_Schema::getRemoteRenderedSchemas($schemaIDs, $post_id);

					if ($renderedSchemas != FALSE) {
						if ( $post_id == get_option( 'page_on_front' ) ) {
							update_option('ps_schema_meta', @$renderedSchemas['meta']);
							update_option('ps_schema_data', @$renderedSchemas['data']);
						} else {
							update_post_meta($post_id, 'ps_schema_meta', @$renderedSchemas['meta']);
							update_post_meta($post_id, 'ps_schema_data', @$renderedSchemas['data']);
						}
					}
				} else {
					if ( $post_id == get_option( 'page_on_front' ) ) {
						update_option('ps_schema_meta', '');
						update_option('ps_schema_data', '');
					} else {
						update_post_meta($post_id, 'ps_schema_meta', FALSE);
						update_post_meta($post_id, 'ps_schema_data', FALSE);						
					}
				}
			}


			/** IDK WHY I USE THIS, BUT IT HAS TO BE HERE */
			if (!wp_is_post_revision($post_id)) {
				remove_action('save_post', ['MXAG_Seo', 'save']);
				add_action('save_post', ['MXAG_Seo', 'save']);
			}

			return $post_id;
		}

		public static function renderSEO($post)
		{
			require_once(XAG_PATH . '/pages/metabox/xag_seo.php');
		}

		public static function formatMetaVariables($meta)
		{
			$tmp = [];
			if (empty($meta) || !$meta) {
				return $tmp;
			}
			foreach ($meta as $key => $value) {
				$tmp[$key] = $value[0];
			}
			return $tmp;
		}

		public static function extract_url_name($url)
		{
			$url = explode("/", $url);
			if (isset($url[sizeof($url) - 2])) {
				$url = $url[sizeof($url) - 2];
			} else {
				$url = $url[0];
			}
			return $url;
		}

		public static function extract_url($post_id)
		{
			$url = get_permalink($post_id);
			$url = str_replace(get_site_url(), '', $url);
			$url = str_replace($_SERVER['HTTP_HOST'], '', $url);
			$url = str_replace('http://', '', $url);
			$url = str_replace('https://', '', $url);
			return $url;
		}

		public static function is_home_static_page()
		{
			return (is_front_page() && 'page' == get_option('show_on_front') && is_page(get_option('page_on_front')));
		}

		public static function is_posts_page()
		{
			return (is_home() && 'page' == get_option('show_on_front'));
		}

		public static function is_home_posts_page()
		{
			return (is_home() && 'posts' == get_option('show_on_front'));
		}

		public static function is_sub_page()
		{
			$object = $GLOBALS['wp_query']->get_queried_object();
			if (is_object($object)) {
				if (is_page() && $object->post_parent > 0) {
					return TRUE;
				}
				return FALSE;
			}

			return FALSE;
		}

		public static function getPostTags($id)
		{
			$post_tags = wp_get_post_tags($id);
			$tags      = [];
			foreach ($post_tags as $tag) {
				$tags[] = $tag->name;
			}
			return join(', ', $tags);
		}

		public static function getPostCategories($id)
		{
			$post_categories = wp_get_post_categories($id);
			$cats            = [];
			foreach ($post_categories as $c) {
				$cat    = get_category($c);
				$cats[] = $cat->name;
			}
			return join(', ', $cats);
		}

		public static function getPostCategoryPrimary($id)
		{
			$category_primary_id = get_post_meta($id, '_category_permalink', TRUE);
			if (!empty($category_primary_id)) {
				$category = get_category($category_primary_id);
				return $category->name;
			} else {
				return '';
			}
		}

		public static function getAllTaxonomies($all = TRUE)
		{
			if ($all == TRUE) {
				return get_taxonomies();
			} else {
				$taxonomies = get_taxonomies([
					'public'   => TRUE,
					'_builtin' => TRUE
				]);
				return $taxonomies;
			}
		}

		public static function getAllPostTypes()
		{
			$post_types = ['post', 'page'];
			foreach (get_post_types(['_builtin' => FALSE, 'public' => TRUE], 'names') as $k => $p) {
				$post_types[] = $p;
			}
			return $post_types;
		}

		public static function getAllPostObjects()
		{
			$post_types = ['post', 'page'];
			foreach (get_post_types(['_builtin' => FALSE, 'public' => TRUE], 'objects') as $k => $p) {
				if (is_object($p)) {
					if (isset($p->name) && isset($p->label)) {
						$post_types[] = [
							'name'  => $p->name,
							'label' => $p->label
						];
					}
				}
			}
			return $post_types;
		}

		public static function getAllCustomPostObjects()
		{
			foreach (get_post_types(['_builtin' => FALSE, 'public' => TRUE], 'objects') as $k => $p) {
				if (is_object($p)) {
					if (isset($p->name) && isset($p->label)) {
						$post_types[] = [
							'name'  => $p->name,
							'label' => $p->label
						];
					}
				}
			}
			return $post_types;
		}

		public static function getAllPosts($ONLY_IDS = FALSE)
		{
			global $wpdb;
			$post_types = self::getAllPostTypes();
			$post_types = "'" . implode("','", $post_types) . "'";
			$out        = $wpdb->get_results("SELECT * FROM $wpdb->posts WHERE post_status = 'publish' AND post_type IN ($post_types)", ARRAY_A);
			if ($ONLY_IDS) {
				$n = [];
				foreach ($out as $p) {
					$n[] = intval($p['ID']);
				}
				$out = $n;
			}
			return $out;
		}
	}

}
