<?php

class MXAG_Update {

	public static function initialize() {
		// Display info of the plugin
		//add_filter( 'plugins_api', array( 'MXAG_Update', 'checkInfo' ), 10, 3 );

		// Cron for checking updates
		add_action('xag_UpdateCheck', array( 'MXAG_Update', 'runUpdater' ));
		if ( ! wp_next_scheduled( 'xag_UpdateCheck' ) ) {
			wp_schedule_event( time(), 'daily', 'xag_UpdateCheck' );
		}

		// Check for updates manually
		add_action('admin_post_xag_check_updates', array('MXAG_Update', 'checkForUpdates'));
		add_action('admin_post_xag_run_updater', array('MXAG_Update', 'runUpdater'));
	}

	public static function checkForUpdates() {
		try {

			$current_version = XAG_CURRENT_VERSION;
			$remote_version  = '0.0.0';

			if (MXAG_Update::checkUpdate($remote_version)) {
				wp_send_json(array('update'=>true, 'remote_version'=>$remote_version, 'current_version'=>$current_version));
			} else {
				wp_send_json(array('update'=>false, 'remote_version'=>$remote_version, 'current_version'=>$current_version));
			}

		} catch (Exception $error) {
			XAG_Init::json('error', 'Plugin error occured. Please pass this down to support: ' . $error->getMessage());
		}
	}

	public static function runUpdater() {
		try {

			if (MXAG_Update::checkUpdate()) {
				$updateURL     = XAG_PANEL_URL . '/update/v3/' . MXAG_Api::getAPIKey();
				$update_folder = XAG_PATH . DIRECTORY_SEPARATOR . 'update' . DIRECTORY_SEPARATOR;

				// Plugin update zip folder
				$zip_file_folder = $update_folder . 'xagio' . DIRECTORY_SEPARATOR;

				// Download the Update
				$zip_file = $update_folder . 'update.zip';
				MXAG_Update::downloadUpdate($zip_file, $updateURL);

				// Patched files
				$files_patched = array();

				// Check if download was a success
				if (file_exists($zip_file)) {

					// Check if the update zip folder already exists and if so, remove it
					if (file_exists($zip_file_folder)) {
						MXAG_Update::removeRecursiveDir($zip_file_folder);
					}

					// Now unzip the update
					if ( class_exists( 'ZipArchive' ) ) {
						$zip = new ZipArchive;
						$res = $zip->open( $zip_file );
						if ( $res === true ) {
							$zip->extractTo( $update_folder );
							$zip->close();
						} else {
							wp_send_json(array(
								'status'  => 'error',
								'message' => 'ZipArchive failed to unzip the update zip.'
							));
							return;
						}
					} else {
						if (!class_exists('PclZip')) {
							require_once ( XAG_PATH . '/ext/pclzip.lib.php' );
						}
						$archive = new PclZip( $zip_file );
						$list = $archive->extract(PCLZIP_OPT_PATH, $update_folder);
						if ($list == 0) {
							wp_send_json(array(
								'status'  => 'error',
								'message' => 'PclZip failed to unzip the update zip.'
							));
							return;
						}
					}

					// Remove the update zip file
					unlink($zip_file);

					// Check if the unzipping was a success
					if (file_exists($zip_file_folder)) {

						// Get all the files
						$files = MXAG_Update::getFilesRecursively($zip_file_folder);

						// Start the patching process
						foreach($files as $file) {
							$new_file = $zip_file_folder . $file;
							$old_file = XAG_PATH . DIRECTORY_SEPARATOR . $file;

							// Check if file exists before anything
							if (!file_exists($new_file)) {
								continue;
							}

							$new_file_md5 = md5_file($new_file);
							$old_file_md5 = 0;

							// Check if folder exists
							$old_dir = dirname($old_file);
							if (!file_exists($old_dir)) {
								@mkdir($old_dir, 0777, true);
							}

							// See if the old file exists before performing md5 check
							if (file_exists($old_file)) {
								$old_file_md5 = md5_file($old_file);
							}

							// If MD5's don't match, overwrite the file
							if ($new_file_md5 !== $old_file_md5) {
								@file_put_contents($old_file, file_get_contents($new_file));
								$files_patched[] = $file;
							}
						}

						// Remove the update folder
						MXAG_Update::removeRecursiveDir($zip_file_folder);

						// Display the success message
						wp_send_json(array(
							'status'  => 'success',
							'message' => 'Successfully updated Xagio V3.',
							'files'   => $files_patched
						));
						return;

					} else {
						wp_send_json(array(
							'status'  => 'error',
							'message' => 'Failed to unzip the update zip.'
						));
						return;
					}

				} else {
					wp_send_json(array(
						'status'  => 'error',
						'message' => 'Failed to download the update zip.'
					));
					return;
				}
			} else {
				wp_send_json(array(
					'status'  => 'error',
					'message' => 'You are already on the latest version.'
				));
				return;
			}

		} catch (Exception $error) {
			XAG_Init::json('error', 'Plugin error occured. Please pass this down to support: ' . $error->getMessage());
		}
	}

	private static function downloadUpdate($location, $file) {
		if (file_exists($location)) {
			@unlink($location);
		}

		$fp = fopen ($location, 'w+');
		$ch = curl_init($file);

		curl_setopt($ch, CURLOPT_TIMEOUT, 60);
		curl_setopt($ch, CURLOPT_FILE, $fp);
		curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);

		curl_exec($ch);
		curl_close($ch);
		fclose($fp);
	}

	private static function getFilesRecursively($dir, $recursive = true, $basedir = '') {
		if ($dir == '') {return array();} else {$results = array(); $subresults = array();}
		if (!is_dir($dir)) {$dir = dirname($dir);} // so a files path can be sent
		if ($basedir == '') {$basedir = realpath($dir).DIRECTORY_SEPARATOR;}

		$files = scandir($dir);
		foreach ($files as $key => $value){
			if ( ($value != '.') && ($value != '..') ) {
				$path = realpath($dir.DIRECTORY_SEPARATOR.$value);
				if (is_dir($path)) { // do not combine with the next line or..
					if ($recursive) { // ..non-recursive list will include subdirs
						$subdirresults = self::getFilesRecursively($path,$recursive,$basedir);
						$results = array_merge($results,$subdirresults);
					}
				} else {
					$path = explode(DIRECTORY_SEPARATOR . 'update' . DIRECTORY_SEPARATOR, $path);
					$path = $path[1];
					$path = str_replace('xagio/', '', $path);
					$subresults[] = $path;
				}
			}
		}
		// merge the subarray to give the list of files then subdirectory files
		if (count($subresults) > 0) {$results = array_merge($subresults,$results);}
		return $results;
	}

	private static function removeRecursiveDir($dir) {
		if( is_dir($dir) ) {
			$objects = array_diff( scandir($dir), array('..', '.') );
			foreach ($objects as $object) {
				$objectPath = $dir."/".$object;
				if( is_dir($objectPath) )
					self::removeRecursiveDir($objectPath);
				else
					unlink($objectPath);
			}
			rmdir($dir);
		}
	}

	public static function checkUpdate(&$remote_version = '0.0.0') {

		// Get the remote version
		$remote_version = MXAG_Update::getRemote_version();

		// If a newer version is available, add the update
		if ( version_compare( XAG_CURRENT_VERSION, $remote_version, '<' ) ) {
			return true;
		}

		return false;
	}

	public static function checkInfo( $false, $action, $arg ) {
		$information = MXAG_Update::getRemote_information();
		return $information;
	}

	public static function getRemote_version() {
		$updateURL = XAG_PANEL_URL . '/update/v3/' . MXAG_Api::getAPIKey();
		$request = wp_remote_post( $updateURL, array( 'body' => array( 'action' => 'version' ) ) );
		if ( ! is_wp_error( $request ) || wp_remote_retrieve_response_code( $request ) === 200 ) {
			return $request['body'];
		}

		return false;
	}

	public static function getRemote_information() {
		$updateURL = XAG_PANEL_URL . '/update/v3/' . MXAG_Api::getAPIKey();
		$request = wp_remote_post( $updateURL, array( 'body' => array( 'action' => 'info' ) ) );
		if ( ! is_wp_error( $request ) || wp_remote_retrieve_response_code( $request ) === 200 ) {
			return unserialize( $request['body'] );
		}

		return false;
	}
}